<?php
/**
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-module/card-manager
 * @subpackage  LivewireComponent
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanModules\CardManager\Livewire\CarmaCards\Modals;


use App\Helpers\LogHelper;
use App\Traits\WithComboBoxInput;
use BplanBase\EntityType\Models\EntityType;
use BplanBase\EntityType\Models\EntityTypeFieldDefinition;
use BplanModules\CardManager\Models\CarmaCardLayout;
use BplanModules\CardManager\Models\CarmaCardType;
use BplanModules\CardManager\Models\CarmaGroup;
use BplanModules\CardManager\Models\CarmaPrinter;
use BplanModules\CardManager\Traits\WithCarmaFields;
use BplanModules\CardManager\Services\CarmaCardService;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Log;
use Livewire\Component;
use Illuminate\Support\Str;


/**
 * @version     1.0.0 / 2025-03-14
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 */
class Add extends Component
{
    use WithComboBoxInput;
    use WithCarmaFields;

    public ?Collection $fieldDefinition;

    protected CarmaCardService $carmaCardService;

    public function rules()
    {
        foreach ($this->fieldDefinition as $fieldDefinitionIndex => $fieldDefinitionField) {
            if($fieldDefinitionField->active && $fieldDefinitionField->mandatory)
            {
                $columnCamelCase = Str::camel($fieldDefinitionField->field->identifier);
                $rules[$columnCamelCase] = 'required';
            }
        }

        return $rules;
    }

    public function boot(CarmaCardService $carmaCardService)
    {
        $this->carmaCardService = $carmaCardService;
    }

    public function mount()
    {
        $this->_setComboBoxData();
        $this->_setFieldDefinition();
    }

    public function render()
    {
        return view('card-manager::livewire.carma-cards.modals.add');
    }

    public function add()
    {
        $this->validate();

        try {
            $attributes = [
                'city' => $this->city,
                'card_layout_id' => $this->getIdsOfSelectedItems('cardLayouts', true),
                'card_type_id' => $this->getIdsOfSelectedItems('cardTypes', true),
                'company' => $this->company,
                'companyCity' => $this->companyCity,
                'company_house_Number' => $this->companyHouseNumber,
                'company_street' => $this->companyStreet,
                'company_zip_code' => $this->companyZipCode,
                'date_of_birth' => $this->dateOfBirth,
                'date_of_entry' => $this->dateOfEntry,
                'department' => $this->department,
                'email' => $this->email,
                'first_name' => $this->firstName,
                'form_of_address' => $this->formOfAddress,
                'function' => $this->function,
                'group_id' => $this->getIdsOfSelectedItems('groups', true),
                'house_number' => $this->houseNumber,
                'identification_number' => $this->identificationNumber,
                'last_name' => $this->lastName,
                'location' => $this->location,
                'street' => $this->street,
                'mobile_phone_number' => $this->mobilePhoneNumber,
                'mobile_phone_number_country_code' => $this->mobilePhoneNumberCountryCode,
                'personnel_number' => $this->personnelNumber,
                'phone_number' => $this->phoneNumber,
                'phone_number_country_code' => $this->phoneNumberCountryCode,
                'printer_id' => $this->getIdsOfSelectedItems('printers', true),
                'zip_code' => $this->zipCode,
            ];

            $this->carmaCardService->create($attributes);

            Log::debug("card created");
            $this->dispatch('card-added');
        } catch (\Throwable $th) {
            LogHelper::logException($th);
        }
    }

    private function _setComboBoxData()
    {
        $this->setComboBoxData('cardLayouts', CarmaCardLayout::all()->map(function ($layout) {
            return [
                'id' => $layout->id,
                'label' => $layout->name,
                // 'tag' => $cardType->labeling,
                'image_url' => $layout->preview_image_url,
            ];
        })->toArray());

        $this->setComboBoxData('cardTypes', CarmaCardType::all()->map(function ($cardType) {
            return [
                'id' => $cardType->id,
                'label' => $cardType->chip_type,
                'tag' => $cardType->labeling,
                'image_url' => asset("img/card-types/{$cardType->identifier}.png"),
            ];
        })->toArray());

        $this->setComboBoxData('groups', CarmaGroup::all()->toArray());

        $this->setComboBoxData('printers', CarmaPrinter::all()->map(function ($printer) {
            return [
                'id' => $printer->id,
                'label' => $printer->name,
                'tag' => $printer->type,
                'image_url' => asset("img/printers/{$printer->identifier}.png"),
            ];
        })->toArray());
    }

    private function _setFieldDefinition()
    {
        $this->fieldDefinition = new Collection();
        $entityType = EntityType::where('identifier', 'CarmaCard.Employee')->first();

        if(isset($entityType))
        {
            $this->fieldDefinition = $entityType->fieldDefinitions()->with('field')->get()->keyBy('field.identifier');
        }
    }
}
