<?php
/**
 * Migration Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-modules/visitor-management
 * @subpackage  Migrations
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <http://www.bplan-solutions.de/>
 * /Δ\
 */


use BplanBase\CodeGenerator\Loaders\CodeGenerator;
use BplanModules\VisitorManagement\Enums\VisitTypeGroupIdentifier;
use BplanModules\VisitorManagement\Enums\VisitTypeIdentifier;
use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;


/**
 * Migration Class
 *
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
return new class extends Migration
{


    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('visit_types');

    } // down()


    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('visit_types', function (Blueprint $Table) {
            $Table->id();
            $Table->boolean('active')->default(1);

            $Table->timestamps();
            $Table->softDeletes('deleted_at');

            $Table->boolean('add_time_slot_booking_link')->default(0)->comment('Flag das anzeigt ob ein Link zur Timeslot-Buchung in die Einladungs-Mail eingefügt werden soll.');
            $Table->json('dependencies')->nullable();
            $Table->enum('identifier', array_column(VisitTypeIdentifier::cases(), 'name'));
            $Table->boolean('internal')->default(0);
            $Table->unsignedInteger('safety_briefing_valid_time')->default(1)->comment('Zeitraum in Monaten für Gültigkeit der Sicherheitsunterweisung.');
            $Table->json('safety_briefings')->nullable();
            $Table->enum('visit_group', array_column(VisitTypeGroupIdentifier::cases(), 'name'));

            $Table->unique('identifier', 'visit_types_UKI_identifier');

            $Table->comment('Technische Daten. Es dürfen weder neue Datensätze hinzugefügt oder vorhandene Datensätze entfernt werden. Ebenso dürfen die Dependencies und die Identifier nicht verändert werden. Lediglich die erweiterten Einstellungen können individuell angepasst werden.');

            new CodeGenerator($Table, [
                'columns' => [
                    'active' => [
                        'sortable' => true,
                    ],
                    'add_time_slot_booking_link' => [
                        'sortable' => true,
                    ],
                    'dependencies' => [
                        'jsonType' => CodeGenerator::JSON_TYPE_OBJECT,
                    ],
                    'identifier' => [
                        'rules' => [
                            'required',
                            'Rule::enum(VisitTypeIdentifier::class)',
                        ],
                        'sortable' => true,
                    ],
                    'safety_briefing_valid_time' => [
                        'sortable' => true,
                    ],
                    'safety_briefings' => [
                        'jsonType' => CodeGenerator::JSON_TYPE_OBJECT,
                    ],
                    'visit_group' => [
                        'rules' => [
                            'required',
                            'Rule::enum(VisitTypeGroupIdentifier::class)',
                        ],
                        'sortable' => true,
                    ],
                ],
                'additionalRelations' => [
                    [
                        /*
                            Model
                            - Methode: appointments(), ReturnType: HasManyThrough, VisitAppointment::class >>> VisitReason::class

                            Schema
                            - HasMany::make('appointments')->type('appointments')

                            Policy
                            - Methode: viewAppointments()

                            Request
                            //

                        */
                        'relationName' => 'appointments',
                        'relationType' => 'HasManyThrough',
                        'resourceName' => 'appointments',
                        'hasManyClass' => 'VisitAppointment::class',
                        'throughClass' => 'VisitReason::class',
                    ],
                ],
                'routing' => [
                    'terminal' => 'readOnly',
                ],
                'uses' => [
                    'jsonapi.request' => [
                        'BplanModules\\VisitorManagement\\Enums\\VisitTypeGroupIdentifier',
                        'BplanModules\\VisitorManagement\\Enums\\VisitTypeIdentifier',
                        'Illuminate\\Validation\\Rule',
                    ],
                ],
            ]);
        });
    } // up()


};
