<?php
/**
 * Procedure Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan/laravel-vega-rpc
 * @subpackage  Procedures
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <http://www.bplan-solutions.de/>
 * /Δ\
 */

declare(strict_types=1);

namespace BplanModules\VisitorManagement\Http\Procedures;


use Bplan\LaravelVegaRpc\Enums\CheckResultCode;
use BplanModules\VisitorManagement\Models\EntryPoint;
use BplanModules\VisitorManagement\Repositories\VisitAppointmentRepository;
use BplanModules\VisitorManagement\Repositories\VisitEntryPointRepository;
use BplanModules\VisitorManagement\Services\VisitorAppointmentService;
use BplanModules\VisitorManagement\Services\VisitorVisitorService;
use Exception;
use Sajya\Server\Annotations\Param;
use Sajya\Server\Annotations\Result;
use Sajya\Server\Http\Request;
use Sajya\Server\Procedure;


/**
 * Definiert RPC-Methoden für die auf dem Terminal auftretenden Events
 *
 * Event-Abfolge
 * 1. receivedAppointment | createdAppointment
 * 2. establishedVoiceConnection
 * 3. canceledVoiceConnection
 * 4. updatedAppointment
 * 5. confirmedSafetyBriefing
 * 6. printedVisitorsLicense
 * 7. finishedWorkflow
 *
 * 8. performedCheckOut
 *
 * -- abortedProcess
 *
 * ? verifiedPhoneNumber
 *
 * @todo        Exception bei fehlender Appointment-ID.
 *
 * @todo        Zur Ermittlung des EntryPoints den RPC-Parameter verwenden.
 *
 * @version     3.0.0 / 2024-12-13
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class TerminalEventProcedure extends Procedure
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * The name of the procedure that is used for referencing.
     *
     * @var     string
     */
    public static string $name = 'terminalevent';


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return 	    EntryPoint
     *
     * @version     1.0.0 / 2024-12-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _determineEntryPoint(Request $Request): EntryPoint
    {
        $requestParams = $Request->getParams()->toArray();

        $entryPoint = $requestParams['entryPoint'];

        if ($entryPoint === null) {
            throw new Exception('Missing required Parameter [entry-point]', CheckResultCode::BadRequest->value);
        }
        if (is_numeric($entryPoint)) {
            $EntryPoint = VisitEntryPointRepository::getById($entryPoint);
        } else {
            $EntryPoint = VisitEntryPointRepository::getModelByToken($entryPoint);
        }
        if ($EntryPoint === null) {
            throw new Exception('Unknwon entry point ['.$entryPoint.']', CheckResultCode::BadRequest->value);
        }
        return $EntryPoint;

    } // _determineEntryPoint()


    /**
     * Aktuell keine Aktion
     *
     * @Param(name="appointmentId", value="required|integer")
     *
     * @Result(name="success", value="required|boolean")
     *
     * @param       Request $Request
     *
     * @return      array
     *
     * @version     1.0.0 / 2024-11-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function abortedProcess(Request $Request): array
    {
        return ['success' => true];

    } // abortedProcess()


    /**
     * Aktuell keine Aktion
     *
     * @Param(name="appointmentId", value="required|integer")
     *
     * @Result(name="success", value="required|boolean")
     *
     * @param       Request $Request
     *
     * @return      array
     *
     * @version     1.0.0 / 2024-11-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function canceledVoiceConnection(Request $Request): array
    {
        return ['success' => true];

    } // canceledVoiceConnection()


    /**
     * Wird aufgerufen sobald der Besucher die Sicherheitsunterweisung bestätigt hat
     *
     * Aktualisiert den Timestamp der Sicherheitsunterweisung beim Visitor.
     *
     * @Param(name="appointmentId", value="required|integer")
     *
     * @Result(name="success", value="required|boolean")
     *
     * @param       Request $Request
     *
     * @return      array
     *
     * @version     1.3.0 / 2024-12-09
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function confirmedSafetyBriefing(Request $Request): array
    {
        $requestParams = $Request->getParams()->toArray();

        $appointmentId = $requestParams['appointmentId'];

        $Appointment = VisitAppointmentRepository::getById($appointmentId);

        (new VisitVisitorService())->setSafetyBriefingConfirmed($Appointment->main_visitor_id);

        return ['success' => true];

    } // confirmedSafetyBriefing()


    /**
     * Wird aufgerufen wenn über das Terminal ein neues Appointment angelegt wurde
     *
     * Setzt den Prozessstatus auf "initialized".
     *
     * @Param(name="appointmentId", value="required|integer")
     *
     * @Result(name="success", value="required|boolean")
     *
     * @param       Request $Request
     *
     * @return      array
     *
     * @version     1.0.0 / 2024-11-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function createdAppointment(Request $Request): array
    {
        $requestParams = $Request->getParams()->toArray();

        $appointmentId = $requestParams['appointmentId'];

        (new VisitAppointmentService())->setProcessStatusInitialized($appointmentId);

        return ['success' => true];

    } // createdAppointment()


    /**
     * Aktuell keine Aktion
     *
     * @Param(name="appointmentId", value="required|integer")
     *
     * @Result(name="success", value="required|boolean")
     *
     * @param       Request $Request
     *
     * @return      array
     *
     * @version     1.0.0 / 2024-11-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function establishedVoiceConnection(Request $Request): array
    {
        return ['success' => true];

    } // establishedVoiceConnection()


    /**
     * Wird aufgerufen wenn der Workflow am Terminal vollständig abgearbeitet wurde
     *
     * Setzt den Prozessstatus auf "checkedIn".
     *
     * @Param(name="appointmentId", value="required|integer")
     *
     * @Result(name="success", value="required|boolean")
     *
     * @param       Request $Request
     *
     * @return      array
     *
     * @version     1.2.0 / 2024-12-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function finishedWorkflow(Request $Request): array
    {
        $EntryPoint = $this->_determineEntryPoint($Request);

        $requestParams = $Request->getParams()->toArray();

        $appointmentId = $requestParams['appointmentId'];

        (new VisitAppointmentService())->setProcessStatusCheckedIn($appointmentId, [
            'check_in_entry_point_id' => $EntryPoint->id,
        ]);

        return ['success' => true];

    } // finishedWorkflow()


    /**
     * Wird aufgerufen, wenn der Besucher am Terminal auscheckt
     *
     * Setzt den Prozessstatus auf "checkedOut".
     * Bei diesem Methodenaufruf kann wahlweise die ID oder das Token des Appointments übergeben
     * werden.
     *
     * @Param(name="appointmentId", value="required|integer, string")
     *
     * @Result(name="success", value="required|boolean")
     *
     * @param       Request $Request
     *
     * @return      array
     *
     * @version     1.4.0 / 2024-12-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function performedCheckOut(Request $Request): array
    {
        $EntryPoint = $this->_determineEntryPoint($Request);

        $requestParams = $Request->getParams()->toArray();

        $appointmentId = $requestParams['appointmentId'];

        if (is_numeric($appointmentId) === false) {
            $Appointment = VisitAppointmentRepository::getModelByToken($appointmentId);

            $appointmentId = $Appointment->id;
        }
        /*
        **  Dieses Event sollte nur im Rahmen von Nicht-Logistik-Besuchen auftreten, weil der
        **  CheckOut nur von einfachen Besuchern am Terminal durchgeführt wird.
        **
        **  @todo   Eventuell sollte hier verhindert werden, dass ein CheckOut für einen Logistik-Prozess
        **          durchgeführt wird.
        */
        (new VisitAppointmentService())->setProcessStatusCheckedOut($appointmentId, [
            'check_out_entry_point_id' => $EntryPoint->id,
        ]);

        return ['success' => true];

    } // performedCheckOut()


    /**
     * Aktuell keine Aktion
     *
     * @Param(name="appointmentId", value="required|integer")
     *
     * @Result(name="success", value="required|boolean")
     *
     * @param       Request $Request
     *
     * @return      array
     *
     * @version     1.0.0 / 2024-11-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function printedVisitorsLicense(Request $Request): array
    {
        return ['success' => true];

    } // printedVisitorsLicense()


    /**
     * Wird aufgerufen sobald erfolgreich ein Appointment ermittelt werden konnte
     *
     * Setzt den Prozessstatus "initialized".
     *
     * @Param(name="appointmentId", value="required|integer")
     *
     * @Result(name="success", value="required|boolean")
     *
     * @todo        Falls es wichtig sein sollte, könnte über einen zusätzlichen Paramter noch mitgegeben werden
     *              auf welchem Weg das Appointment ermittelt wurde (determinedBy QrCode | Token | OrderNumber).
     *
     * @param       Request $Request
     *
     * @return      array
     *
     * @version     1.0.0 / 2024-12-09
     * @history     TerminalEventProcedure::scanedQrCode(), 1.0.0 / 2024-11-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function receivedAppointment(Request $Request): array
    {
        $requestParams = $Request->getParams()->toArray();

        $appointmentId = $requestParams['appointmentId'];

        (new VisitAppointmentService())->setProcessStatusInitialized($appointmentId);

        return ['success' => true];

    } // receivedAppointment()


    /**
     * Aktuell keine Aktion
     *
     * @Param(name="appointmentId", value="required|integer")
     *
     * @Result(name="success", value="required|boolean")
     *
     * @param       Request $Request
     *
     * @return      array
     *
     * @version     1.0.0 / 2024-11-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function updatedAppointment(Request $Request): array
    {
        return ['success' => true];

    } // updatedAppointment()


    /**
     * Aktuell keine Aktion
     *
     * @Param(name="appointmentId", value="required|integer")
     *
     * @Result(name="success", value="required|boolean")
     *
     * @param       Request $Request
     *
     * @return      array
     *
     * @version     1.0.0 / 2024-11-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function verifiedPhoneNumber(Request $Request): array
    {
        return ['success' => true];

    } // verifiedPhoneNumber()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class TerminalEventProcedure extends Procedure {}
