<?php
/**
 * Livewire Component Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-modules/visitor-management
 * @subpackage  LivewireComponents
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanModules\VisitorManagement\Livewire\VisitAppointments\Modals;


use BplanBase\Globals\Enums\LogLevel;
use BplanBase\Globals\Services\LogService;
use BplanBase\GlobalComponents\Traits\WithComboBoxInput;
use BplanModules\VisitorManagement\Enums\AppointmentSteps;
use BplanBase\Globals\Enums\RequireStatus;
use BplanModules\VisitorManagement\Enums\VisitTypeGroupIdentifier;
use BplanModules\VisitorManagement\Enums\VisitTypeIdentifier;
use BplanModules\VisitorManagement\Models\VisitAppointment;
use BplanModules\VisitorManagement\Models\VisitEmployee;
use BplanModules\VisitorManagement\Models\VisitEquipmentItem;
use BplanModules\VisitorManagement\Models\VisitVisitor;
use BplanModules\VisitorManagement\Models\VisitReason;
use BplanModules\VisitorManagement\Repositories\VisitEmployeeRepository;
use BplanModules\VisitorManagement\Repositories\VisitEquipmentItemRepository;
use BplanModules\VisitorManagement\Repositories\VisitReasonRepository;
use BplanModules\VisitorManagement\Services\VisitAppointmentService;
use BplanModules\VisitorManagement\Services\VisitAppointmentEquipmentItemService;
use BplanModules\VisitorManagement\Traits\AppointmentPagination;
use BplanModules\VisitorManagement\Traits\WithLicensePlateInput;
use Illuminate\Database\Eloquent\Collection;
use Laravel\Jetstream\InteractsWithBanner;
use Livewire\Attributes\On;
use Livewire\Component;
use stdClass;
use Illuminate\Support\Str;


/**
 * Livewire Component Class
 *
 * @version     1.1.0 / 2025-04-29
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class Edit extends Component
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    use InteractsWithBanner;
    use WithLicensePlateInput;
    use WithComboBoxInput;
    use AppointmentPagination;


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     */
    // #[Validate('required')]
    public bool $isSpontaneousVisit = false;


    /**
     *
     */
    // #[Validate('required')]
    public ?string $subject;


    /**
     *
     */
    // #[Validate('required')]
    public ?string $orderNumber = null;


    /**
     *
     */
    // #[Validate('exists:visit_reasons,id')]
    public ?int $visitReasonId = null;


    /**
     *
     */
    // #[Validate('exists:visit_reasons,id')]
    public ?string $bulkVisitTypeIdentifier = null;


    /**
     *
     */
    // #[Validate('exists:employees,id')] // nullable|numeric|
    public ?int $mainEmployeeId = null;


    /**
     *
     */
    // #[Validate('required')] // nullable|numeric|
    public ?string $licensePlateFormatted = ''; //* Nur wenn driveOnAreaPermitted = true || VisitType == AccessAuthorization


    /**
     *
     */
    public ?int $organizerEmployeeId = null;


    /**
     *
     */
    // #[Validate('required')]
    public ?array $visitors;


    /**
     *
     */
    // #[Validate('exists:visitors,id')] // nullable|numeric|
    public ?int $mainVisitorId = null;


    /**
     *
     */
    // #[Validate('required')]
    public ?string $startDate;


    /**
     *
     */
    // #[Validate('required')]
    public ?string $startTime;


    /**
     *
     */
    // #[Validate('required')]
    public ?string $endDate;


    /**
     *
     */
    // #[Validate('required')]
    public ?string $endTime;


    /**
     *
     */
    // #[Validate('required')]
    public bool $driveOnAreaPermitted = false;


    /**
     *
     */
    // #[Validate('required')]
    public bool $withMinors = false;


    /**
     *
     */
    // #[Validate('required')]
    public bool $requireEscort = false;


    /**
     *
     */
    // #[Validate('required')]
    public bool $active = true;


    /**
     *
     */
    public ?bool $internalVisit = false;


    /**
     *
     */
    public ?string $appointmentBegin;

    /**
     *
     */
    // #[Validate('after:appointmentBegin')]
    public ?string $appointmentEnd;


    /**
     *
     */
    public ?array $appointmentVisitors = [];


    /**
     *
     */
    public ?VisitAppointment $appointment;


    /**
     *
     */
    // Wird für WithComboBoxInput benötigt
    public ?array $comboBoxData;

    /**
     *
     */
    public ?array $filteredComboBoxData;


    /**
     *
     */
    public bool $isBulkVisit = false;


    /**
     *
     */
    public ?Collection $employees;

    /**
     *
     */
    public ?Collection $_allVisitors;

    /**
     *
     */
    public ?Collection $visitReasons;

    /**
     *
     */
    public ?stdClass $visitReasonLabels;

    /**
     *
     */
    public ?array $visitReasonsByVisitType;

    /**
     *
     */
    public ?Collection $equipmentItems;

    /**
     *
     */
    public ?array $appointmentEquipmentItems = [];


    /**
     *
     */
    protected LogService $logService;


    /**
     *
     */
    protected VisitAppointmentService $appointmentService;


    /**
     *
     */
    protected VisitAppointmentEquipmentItemService $appointmentEquipmentItemService;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function rules()
    {
        $rules = [];

        if($this->appointmentStep >= AppointmentSteps::General->value)
        {
            $rules['isSpontaneousVisit'] = 'required';
            $rules['subject'] = 'required';
            $rules['appointmentBegin'] = 'required';
            $rules['appointmentEnd'] = 'required|after:appointmentBegin';
            $rules['visitReasonId'] = 'required|exists:visit_reasons,id';
        }

        if($this->appointmentStep >= AppointmentSteps::Visitors->value)
        {
            $this->_setMainEmployee();
            $this->_setMainVisitor();
            $this->_setOrganizer();
            $this->_setVisitors();

            //* Verantwortlicher Mitarbeiter immer optional - so von Avient gewünscht
            // $rules['mainEmployeeId'] = 'nullable|required_if:isSpontaneousVisit,false|exists:employees,id';
            // $rules['mainVisitorId'] = 'nullable|required_if:isSpontaneousVisit,false|exists:visitors,id';
            //$rules['licensePlateFormatted'] = 'nullable|required_if:driveOnAreaPermitted,true';

            $visitReason = $this->visitReasons->where('id',$this->visitReasonId)->first();
            $this->visitReasonLabels = $visitReason?->labels;
            if(isset($this->visitReasonId) && isset($visitReason?->visitType?->dependencies))
            {
                $visitReason = $this->visitReasons->where('id',$this->visitReasonId)->first();

                foreach ($visitReason->visitType->dependencies as $column => $required) {
                    if($required >= RequireStatus::Required->value)
                    {
                        // if($column == 'license_plate_formatted')
                        // {
                        //     if($this->isSpontaneousVisit)
                        //     {
                        //        continue;
                        //     }

                        //     $column = 'licensePlates';
                        // }

                        switch ($column) {
                            case 'license_plate_formatted':
                                if ($this->driveOnAreaPermitted === false || $this->isSpontaneousVisit) {
                                    continue 2;
                                }
                                break;
                            case 'main_visitor_id':
                                if ($this->isBulkVisit == false) {
                                    $rules['mainVisitorId'] = 'nullable|required_if:isSpontaneousVisit,false|exists:visitors,id';
                                }
                                break;

                            default:
                                // ? In den dependencies sind die spalten in snake case.
                                // ? So wird aus z.B. "main_visitor_id" => "mainVisitorId"
                                // ? Welches dann der Variablen namen in dieser Component entspricht.
                                $columnCamelCase = Str::camel($column);

                                $rules[$columnCamelCase] = 'required';
                                break;
                        }

                    }
                }

            }
        }

        return $rules;

    } // rules()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function boot(LogService $logService, VisitAppointmentService $appointmentService, VisitAppointmentEquipmentItemService $appointmentEquipmentItemService)
    {
        $this->logService = $logService;
        $this->appointmentService = $appointmentService;
        $this->appointmentEquipmentItemService = $appointmentEquipmentItemService;

    } // boot()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function mount()
    {
        $this->employees = VisitEmployeeRepository::getAll();
        $this->_allVisitors = VisitVisitor::where('verified', 1)->where('anonymized', 0)->get();
        $this->equipmentItems = VisitEquipmentItemRepository::getAll();
        $this->appointmentEquipmentItems = $this->equipmentItems->mapWithKeys(function ($item) {
            $item['equipment_item_id'] = $item->id;
            $item['amount'] = 0;
            return [$item->id => $item];
        })->toArray() ?? [];

        /*
        **  VisitReasons ermitteln und dabei alle Reasons ignorieren deren VisitType "internal = 1" ist
        **  oder die selbst "internal = 1" sind. */
        // dd($this->appointment->visitType->visit_group);
        $this->visitReasons = VisitReasonRepository::getAll();

        // VisitReason::whereRelation('visitType', 'internal', 0)
        //     ->whereRelation('visitType', 'active', 1)
        //     ->where('internal', '=', 0)
        //     ->where('active', '=', 1)
        //     ->orderBy('labels->'.app()->getLocale())
        //     ->get();
        //->mapWithKeys(function ($item) {
        //    return [$item['id'] => $item];
        //});

        // $this->visitReasonsByVisitType = [];
        foreach ($this->visitReasons as $key => $visitReason) {
            $this->visitReasonsByVisitType[$visitReason->visitType->identifier][] = $visitReason;
        }

        $this->setupComboBoxes();

    } // mount()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function render()
    {
        return view('visitor-management::livewire.visit-appointments.modals.edit');

    } // render()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function resetModal()
    {
        //$this->clearLicensePlates();
        $this->licensePlateFormatted = null;
        $this->setupComboBoxes();
        $this->comboBoxDataDeselectAll('visitors');
        $this->resetErrorBag();
        // $this->appointment = null;

        // $this->isSpontaneousVisit = false;
        // $this->subject = null;
        // $this->order_number = null;
        // $this->mainVisitorId = null;
        // $this->visitReasonId = null;
        // $this->mainEmployeeId = null;
        // $this->licensePlate = null;
        // $this->appointmentBegin = null;
        // $this->appointmentEnd = null;
        // $this->driveOnAreaPermitted = false;
        // $this->withMinors = false;
        // $this->requireEscort = false;

        // $this->active = true;

    } // resetModal()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function removeVisitor($visitorIndex)
    {
        unset($this->appointmentVisitors[$visitorIndex]);

    } // removeVisitor()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    #[On('visitors-upload-finished')]
    public function onVisitorsUploaded()
    {
        $this->appointment = VisitAppointment::findOrFail($this->appointment->id);
        $this->appointmentVisitors = $this->appointment->visitors;

    } // onVisitorsUploaded())


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    #[On('show-edit-appointment-modal')]
    public function initAppointment(VisitAppointment $appointment)
    {
        $this->appointment = $appointment;

        $this->isSpontaneousVisit = $appointment->spontaneous_visit ?? false;
        $this->subject = $appointment->subject;
        $this->orderNumber = $appointment->order_number;

        $mainVisitorIndex = $this->_allVisitors->search(function ($visitor, $key) use ($appointment) {
            return $appointment->main_visitor_id == $visitor->id;
        });

        if(is_numeric($mainVisitorIndex))
        {
            $this->comboBoxData['mainVisitors'][$mainVisitorIndex]['isSelected'] = true;
        }

        $mainEmployeeIndex = $this->employees->search(function ($employee, $key) use ($appointment) {
            return $appointment->main_employee_id == $employee->id;
        });

        if(is_numeric($mainEmployeeIndex))
        {
            $this->comboBoxData['mainEmployees'][$mainEmployeeIndex]['isSelected'] = true;
        }

        $organizerEmployeeIndex = $this->employees->search(function ($employee, $key) use ($appointment) {
            return $appointment->organizer_employee_id == $employee->id;
        });

        if(is_numeric($organizerEmployeeIndex))
        {
            $this->comboBoxData['organizers'][$organizerEmployeeIndex]['isSelected'] = true;
        }

        $this->visitReasonId = $appointment->bulk_informations->visit_reason_id ?? $appointment->visit_reason_id;
        $this->bulkVisitTypeIdentifier = $appointment->bulk_informations->visit_type_identifier ?? null;
        $this->isBulkVisit = $appointment->visitType->identifier == VisitTypeIdentifier::BulkAppointment->value;
        // $this->mainEmployeeId = $appointment->main_employee_id;
        $this->licensePlateFormatted = $appointment->license_plate_formatted;

        $this->driveOnAreaPermitted = $appointment->drive_on_area_permitted;
        $this->withMinors = $appointment->with_minors;
        $this->requireEscort = $appointment->require_escort;
        $this->active = $appointment->active;
        $this->internalVisit = $appointment->internal_visit;

        $this->startDate = $appointment->valid_from->toDateString();
        $this->startTime = $appointment->valid_from->toTimeString();
        $this->endDate = $appointment->valid_until->toDateString();
        $this->endTime = $appointment->valid_until->toTimeString();

        $this->appointmentBegin = $appointment->valid_from->toDateTimeString();
        $this->appointmentEnd = $appointment->valid_until->toDateTimeString();

        if($appointment->visitType->identifier != VisitTypeIdentifier::GroupVisit->value)
        {
            foreach ($this->appointment->visitors ?? [] as $key => $visitor) {
                if(isset($visitor->id) && $visitor->id > 0)
                {
                    $this->setComboBoxDataItemSelectedByAttr('visitors', 'id', $visitor->id);
                }
            }
        } else
        {
            $this->appointmentVisitors = (array) $this->appointment?->visitors;
        }


        foreach ($this->appointment->equipmentItems as $key => $appointmentItem) {
            $this->appointmentEquipmentItems[$appointmentItem->id]['equipment_item_id'] = $appointmentItem->id;
            $this->appointmentEquipmentItems[$appointmentItem->id]['amount'] = $appointmentItem->pivot->amount ?? 0;
        }

        $this->dispatch('init-tippyjs');

    } // initAppointment()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function updated($property)
    {
        switch ($property) {
                case 'startDate':
                case 'startTime':
                    try {
                        $this->appointmentBegin = $this->startDate . 'T' . $this->startTime;
                        $this->appointmentEnd = $this->endDate . 'T' . $this->endTime;
                    } catch (\Throwable $th) {
                        //throw $th;
                    }
                case 'endDate':
                case 'endTime':
                    try {
                        $this->appointmentEnd = $this->endDate . 'T' . $this->endTime;
                    } catch (\Throwable $th) {
                        //throw $th;
                    }
                    break;
                case 'mainVisitorId':
                    // TODO CHECK IF DATENSATZ VOLLSTÄDNIG
                    break;
                case 'internalVisit':
                    if($this->internalVisit)
                        $this->requireEscort = false;
                    break;
                case 'requireEscort':
                    if($this->requireEscort)
                        $this->internalVisit = false;
                    break;
                case 'visitReasonId':
                    $visitReason = $this->visitReasons->where('id', $this->visitReasonId)->first();
                    if (isset($visitReason)) {
                        if ($visitReason->visitType->visit_group == VisitTypeGroupIdentifier::Logistics->value) {
                            $this->driveOnAreaPermitted = true;
                        }
                    }
                    break;

            default:
                # code...
                break;
        }
    } // updated()


    /**
     *
     * @version     1.1.0 / 2025-04-29
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    #[On('save-edit-appointment')]
    public function editAppointment(bool $notifyParticipants = false)
    {
        $this->_setMainEmployee();
        $this->_setMainVisitor();

        $this->validate();

        try {
            $selectedVisitors = array_filter(
                array_map(function ($item) {
                    if (isset($item['isSelected']) && $item['isSelected']) {
                        return $item['id'];
                    }
                }, $this->comboBoxData['visitors'])
            );


            // Alle visitors mit ids aussortieren
            // $visitors = (array) $this->appointment->visitors ?? [];
            // foreach ($visitors as $key => $visitor) {
            //     if(isset($visitor->id) && $visitor->id > 0)
            //     {
            //         unset($visitors[$key]);
            //     }
            // }

            $attributes = [
                'spontaneous_visit' => $this->isSpontaneousVisit,
                'subject' => $this->subject,
                'order_number' => $this->orderNumber,
                'main_visitor_id' => $this->mainVisitorId,
                'visit_reason_id' => $this->visitReasonId,
                'organizer_employee_id' => $this->organizerEmployeeId,
                'main_employee_id' => $this->mainEmployeeId,
                // 'license_plate' => $this->licensePlate ?? null,
                'valid_from' => $this->appointmentBegin,
                'valid_until' => $this->appointmentEnd,
                'drive_on_area_permitted' => $this->driveOnAreaPermitted,
                'with_minors' => $this->withMinors,
                'require_escort' => $this->requireEscort,
                'active' => $this->active,
                'visitors' => $this->appointmentVisitors,
                'internal_visit' => $this->internalVisit,
                'license_plate_formatted' => $this->licensePlateFormatted,
            ];

            if($this->appointment->visitType->identifier == VisitTypeIdentifier::BulkAppointment->value)
            {
                $this->appointmentService->updateBulk($this->appointment, $attributes, $selectedVisitors, $notifyParticipants);
            }
            else
            {
                $this->appointmentService->update($this->appointment, $attributes, $selectedVisitors, $notifyParticipants);
            }


            // Main Visitor zusätzlich verlinken
            // $visitorsIds[] = ['id' => $this->mainVisitorId];

            $this->logService->log(LogLevel::Info, "Appointment {$this->subject} edited by " . auth()->user()->name . ' (' . auth()->user()->email . ')');

            $this->dispatch('init-tippyjs');
            $this->dispatch('appointment-edited');
            $this->banner(__('visitor-management::visit-appointments.appointment_saved_successfully', [
                'subject' => $this->subject
            ]));
        } catch (\Throwable $th) {
            $this->logService->log(LogLevel::Error, 'Appointments/Modals/editAppointment error: ' . $th->getMessage() . ', file:' . $th->getFile() . ', line:' .  $th->getLine());
            $this->dangerBanner('Error when editing appointment. Check the logs for more information.');
        }
    } // editAppointment()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function saveEquipmentItems()
    {
        try {
            $this->appointmentEquipmentItemService->saveOrDeleteMultiple($this->appointmentEquipmentItems, $this->appointment->id);
            $this->logService->log(LogLevel::Info, "Appointment {$this->subject} equipment items edited by " . auth()->user()->name . ' (' . auth()->user()->email . ')');
            $this->banner(__('visitor-management::visit-appointments.appointment_saved_successfully', [
                'subject' => $this->subject
            ]));
        } catch (\Throwable $th) {
            $this->logService->log(LogLevel::Error, 'Appointments/Modals/saveEquipmentItems error: ' . $th->getMessage() . ', file:' . $th->getFile() . ', line:' .  $th->getLine());
            $this->dangerBanner('Error when editing equipment items. Check the logs for more information.');
        }
    } // saveEquipmentItems()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function deleteAppointment()
    {
        try
        {
            $this->appointmentService->delete($this->appointment->id);

            $this->dispatch('appointment-deleted');
        } catch (\Throwable $th) {
            $this->logService->log(LogLevel::Error, 'Appointments/Modals/deleteAppointment error: ' . $th->getMessage() . ', file:' . $th->getFile() . ', line:' .  $th->getLine());
            $this->dangerBanner('Error when deleting appointment. Check the logs for more information.');
        }
    } // deleteAppointment()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function setupComboBoxes()
    {
        $this->comboBoxData['visitors'] = $this->_allVisitors->map(function ($visitor) {
            $visitorArray = $visitor->getAttributes();
            $visitorArray['label'] = "{$visitor->last_name}, {$visitor->first_name}";
            $visitorArray['tag'] = "{$visitor->email}";

            return $visitorArray;
        })->toArray();

        $this->comboBoxData['mainVisitors'] = $this->_allVisitors->map(function ($visitor) {
            $visitorArray = $visitor->getAttributes();
            $visitorArray['label'] = "{$visitor->last_name}, {$visitor->first_name}";
            $visitorArray['tag'] = "{$visitor->email}";

            return $visitorArray;
        })->toArray();

        $this->comboBoxData['mainEmployees'] = $this->employees->map(function ($employee) {
            $visitorArray = $employee->getAttributes();
            $visitorArray['label'] = "{$employee->last_name}, {$employee->first_name}";
            $visitorArray['tag'] = "{$employee->email}";

            return $visitorArray;
        })->toArray();

        $this->comboBoxData['organizers'] = $this->employees->whereNotNull('email')->map(function ($employee) {
            $visitorArray = $employee->getAttributes();
            $visitorArray['label'] = "{$employee->last_name}, {$employee->first_name}";
            $visitorArray['tag'] = "{$employee->email}";

            return $visitorArray;
        })->toArray();

        $this->filteredComboBoxData = $this->comboBoxData;

    } // setupComboBoxes()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _setMainEmployee()
    {
        $mainEmployee = array_filter(
            array_map(function ($item) {
                if (isset($item['isSelected']) && $item['isSelected']) {
                    return $item['id'];
                }
            }, $this->comboBoxData['mainEmployees'])
        );

        if(isset($mainEmployee) && !empty($mainEmployee))
        {
            $this->mainEmployeeId = $mainEmployee[array_key_first($mainEmployee)];
        }
    } // _setMainEmployee()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _setMainVisitor()
    {
        $mainVisitor = array_filter(
            array_map(function ($item) {
                if (isset($item['isSelected']) && $item['isSelected']) {
                    return $item['id'];
                }
            }, $this->comboBoxData['mainVisitors'])
        );

        if(isset($mainVisitor) && !empty($mainVisitor))
        {
            $this->mainVisitorId = $mainVisitor[array_key_first($mainVisitor)];
        }
    } // _setMainVisitor()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _setOrganizer()
    {
        $organizerEmployee = array_filter(
            array_map(function ($item) {
                if (isset($item['isSelected']) && $item['isSelected']) {
                    return $item['id'];
                }
            }, $this->comboBoxData['organizers'])
        );

        if (isset($organizerEmployee) && !empty($organizerEmployee)) {
            $this->organizerEmployeeId = $organizerEmployee[array_key_first($organizerEmployee)];
        }
    } // _setOrganizer()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _setVisitors()
    {
        $visitors = array_filter(
            array_map(function ($item) {
                if (isset($item['isSelected']) && $item['isSelected']) {
                    return $item['id'];
                }
            }, $this->comboBoxData['visitors'])
        );

        if (isset($visitors) && !empty($visitors)) {
            $this->visitors = $visitors;
        }
    } // _setVisitors()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class Edit extends Component {}
