<?php

namespace BplanModules\VisitorManagement\Livewire\VisitLogisticQueue\Modals;

use BplanBase\Globals\Enums\LogLevel;
use BplanBase\Globals\Services\LogService;
use BplanModules\VisitorManagement\Enums\LocationType;
use BplanModules\VisitorManagement\Models\VisitAppointment;
use BplanModules\VisitorManagement\Models\VisitLocation;
use BplanModules\VisitorManagement\Models\VisitLocationType;
use BplanModules\VisitorManagement\Services\VisitAppointmentService;
use Illuminate\Database\Eloquent\Collection;
use Laravel\Jetstream\InteractsWithBanner;
use Livewire\Attributes\On;
use Livewire\Attributes\Validate;
use Livewire\Component;

class LetInVisitor extends Component
{
    use InteractsWithBanner;

    #[Validate('required|exists:locations,id')]
    public ?int $locationId;

    public ?Collection $locations;
    public ?VisitAppointment $appointment;

    protected LogService $logService;
    protected VisitAppointmentService $appointmentService;

    public function boot(LogService $logService, VisitAppointmentService $appointmentService)
    {
        $this->logService = $logService;
        $this->appointmentService = $appointmentService;
    }

    public function mount()
    {
        $loadingRampId = VisitLocationType::where('identifier', LocationType::LoadingRamp)->first()->id;
        $this->locations = VisitLocation::where('location_type_id', $loadingRampId)->get();
        $this->locationId = $this->locations[0]?->id ?? null;
    }

    public function render()
    {
        return view('visitor-management::livewire.visit-logistic-queue.modals.let-in-visitor');
    }

    #[On('show-let-in-logistic-modal')]
    public function initAppointment(VisitAppointment $appointment)
    {
        $this->appointment = $appointment;

        $this->dispatch('init-tippyjs');
    }

    public function letInLogistic()
    {
        $this->validate();

        try {
            $this->appointmentService->assignLoadingRamp($this->appointment->id, $this->locationId);
            $this->appointment = VisitAppointment::findOrFail($this->appointment->id);
            $this->logService->log(LogLevel::Info, "Logistic were let in by " . auth()->user()->name . ' (' . auth()->user()->email . ')');

            $this->banner(trans('visitor-management::visit-logistic-queue.banner.let-in-to-location', [
                'logistic' => "{$this->appointment->mainVisitor->last_name}, {$this->appointment->mainVisitor->first_name}",
                'location' => "{$this->appointment->loadingRamp->labels->{app()->getLocale()} }",
            ]));

            $this->dispatch('init-tippyjs');
            $this->dispatch('logistic-queue-edited');
        } catch (\Throwable $th) {
            $this->logService->log(LogLevel::Error, 'LogisticQueue/Modals/LetInModal saveAppointment: ' . $th->getMessage() . ', file:' . $th->getFile() . ', line:' .  $th->getLine());
            $this->dangerBanner('Error when letting in logistic. Check the logs for more information.');
        }
    }

    public function resetModal()
    {
        $this->appointment = null;

        $this->locationId = $this->locations[0]?->id ?? null;
    }
}
