<?php
/**
 * Notification Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-modules/visitor-management
 * @subpackage  Notifications
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanModules\VisitorManagement\Notifications;


use App\Repositories\UserRepository;
use BplanBase\Globals\Enums\ModelEvent;
use BplanComponents\ICS\ICS\Event;
use BplanComponents\ICS\ICS\ICSFile;
use BplanModules\VisitorManagement\Enums\VisitTypeIdentifier;
use BplanModules\VisitorManagement\Models\VisitAppointment;
use BplanModules\VisitorManagement\Notifications\Messages\PHPMailerMessage;
use BplanModules\VisitorManagement\Repositories\VisitVisitorRepository;
use BplanModules\VisitorManagement\Repositories\VisitTypeRepository;
use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\Log;


/**
 * Notification Class
 *
 * Benachrichtigung an einen Mitarbeiter (verantwortlicher Mitarbeiter) über die Anlage eines
 * Termins (per E-Mail) mit angehängter ics-Datei.
 *
 * @version     2.3.0 / 2025-03-18
 * @history     EmployeeAppointmentCreated, 1.3.0 / 2025-01-12
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class EmployeeAppointmentSaved extends Notification
{

/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    use Queueable;


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     VisitAppointment $_Appointment
     */
    private $_Appointment;


    /**
     * @var     ModelEvent $_Event
     */
    private $_Event;


    /**
     * @var     VisitVisitor $_Visitor
     */
    private $_Visitor;


    /**
     * @var     VisitType $_VisitType
     */
    private $_VisitType;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Create a new notification instance.
     *
     * @param       VisitAppointment $Appointment
     *
     * @param       ModelEvent $Event
     *
     * @version     2.3.0 / 2025-01-19
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(VisitAppointment $Appointment, ModelEvent $Event)
    {
        $this->_Appointment = $Appointment;
        $this->_Event = $Event;

        $this->_VisitType = VisitTypeRepository::getByVisitReason($Appointment->visit_reason_id);
        /*
        **  Appointments vom VisitType "BulkAppointment" haben keinen Hauptbesucher. */
        if ($this->_VisitType->identifier !== VisitTypeIdentifier::BulkAppointment->value) {
            $this->_Visitor = VisitVisitorRepository::getById($this->_Appointment->main_visitor_id);
        }
    } // __construct()


    /**
     *
     * @param       VisitEmployee $Notifiable
     *
     * @param       null|string & $icsContents
     *
     * @return 	    string
     *
     * @version     1.1.0 / 2025-01-19
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _buildIcsFile(object $Notifiable, null|string & $icsContents = null): string
    {
        $lang = 'de';

        $company = config('project.company');

        $CreateUser = UserRepository::getById($this->_Appointment->create_user_id);
        /*
        **  Event für ics-Datei zusammenstellen.
        **
        **  Da die verwendete ICS-Klasse keine Möglichkeit bietet eine Timezone zu definieren, muss die
        **  Timezone für Beginn und Ende nach "UTC" geändert werden. */
        $eventDetails = [
            'description' => $this->_Appointment->subject,
            'dtstart' => $this->_Appointment->valid_from->setTimezone('UTC')->format('Y-m-d H:i'),
            'dtend' => $this->_Appointment->valid_until->setTimezone('UTC')->format('Y-m-d H:i'),
            'location' => $company['full_name'].', '.$company['address'],
            'sequence' => 0,
            'summary' => $this->_Appointment->subject,
            'uid' => $this->_Appointment->uuid,
        ];
        $Event = new Event($eventDetails);
        /*
        **  Organizer ist immer der erstellende User und der erste Attendee ist der Mitarbeiter. */
        $Event->setOrganizer($CreateUser->getFullName(), $CreateUser->email);
        $Event->addAttendee($Notifiable->getFullName(), $Notifiable->email, accepted: true);
        /*
        **  Dieser Fall tritt ein wenn es sich um ein Appointment vom VisitType "BulkAppointment" handelt. In diesem
        **  gibt es keinen Hauptbesucher. Stattdessen sind die Basisdaten der Besucher im visitors-Feld Fall des
        **  Appointments gespeichert.
        **  Alternativ könnten die zugehörigen Besucher auch aus der Datenbank gelesen werden, indem die Hauptbesucher
        **  aller Einzeltermine ermittelt werden. Für den konkreten Fall sind die beim Appointment gespeicherten Daten
        **  aber vollkommen ausreichend. */
        if ($this->_VisitType->identifier === VisitTypeIdentifier::BulkAppointment->value) {
            foreach ($this->_Appointment->visitors as $Visitor) {
                $Event->addAttendee($Visitor->first_name.' '.$Visitor->last_name, $Visitor->email ?? trans('notifications.no-mail'));
            }
        } else {
            $Event->addAttendee($this->_Visitor->getFullName(), $this->_Visitor->email ?? trans('notifications.no-mail'));
        }
        /*
        **  ics-Datei erstellen und speichern. */
        $IcsFile = new ICSFile();

        $IcsFile->addEvent($Event);
        $IcsFile->setLanguage($lang);

        $storagePath = storage_path('app/tmp');

        if (!file_exists($storagePath)) {
            mkdir($storagePath, 0777, true);
        }
        $icsFile = $storagePath.'/'.$this->_Appointment->uuid.'.ics';
        $icsContents = $IcsFile->toString();

        file_put_contents($icsFile, $icsContents);

        return $icsFile;

    } // _buildIcsFile()


    /**
     * Get the array representation of the notification.
     *
     * @param       VisitEmployee $Notifiable
     *
     * @return      array<string, mixed>
     *
     * @version     1.0.0 / 2024-11-30
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function toArray(object $Notifiable): array
    {
        return [
            //
        ];
    } // toArray()


    /**
     * Get the mail representation of the notification.
     *
     * @param       VisitEmployee $Notifiable
     *
     * @throws      Exception
     *
     * @return      MailMessage
     *
     * @version     1.6.0 / 2025-03-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function toPHPMailer(object $Notifiable): PHPMailerMessage
    {
        $lang = 'de';

        $lc_dateTimeFormat = trans('globals::global.datetime.format-+', [], $lang);

        $icsFile = $this->_buildIcsFile($Notifiable, $icsContents);

        $Message = new PHPMailerMessage();

        $Message->addRecipient($Notifiable->email, $Notifiable->getFullName())
            ->addAttachment($icsFile, 'invite.ics')
            ->setIcal($icsContents)
            ->setSubject(trans('notifications.appointment-saved.mail.employee.subject', [
                    'date' => $this->_Appointment->valid_from->format($lc_dateTimeFormat)
                ], $lang)
            );

        Log::notice('Sending employee '.$this->_Event->value.'-notification to "'.$Notifiable->getFullName().'" ('.$Notifiable->email.') in '.__METHOD__.'().');

        return $Message;

    } // toPHPMailer()


    /**
     * Get the notification's delivery channels.
     *
     * @param       VisitEmployee $Notifiable
     *
     * @return      array<int, string>
     *
     * @version     1.0.0 / 2024-11-30
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function via(object $Notifiable): array
    {
        return [\BplanModules\VisitorManagement\Channels\PHPMailerChannel::class];

    } // via()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class EmployeeAppointmentSaved extends Notification {}
