<?php
/**
 * Migration Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-modules/visitor-management
 * @subpackage  Migrations
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */


use BplanBase\CodeGenerator\Loaders\CodeGenerator;
use BplanModules\VisitorManagement\Enums\ProcessStatus;
use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;


/**
 * Migration Class
 *
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
return new class extends Migration
{


    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('visit_appointments');

    } // down()


    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('visit_appointments', function (Blueprint $Table) {
            $Table->id();
            $Table->uuid('uuid');
            $Table->boolean('active')->default(1);

            $Table->timestamps();
            $Table->softDeletes('deleted_at');

            $Table->foreignId('bulk_appointment_id')->nullable()->comment('Bei Einzelbesuchen zu einem Massentermin, die ID des Massentermins.')->constrained(table: 'visit_appointments', indexName: 'visit_appointments_FK_bulk_appointment_id');
            $Table->json('bulk_informations')->nullable()->comment('Erweiterte Informationen zu einem Massentermin.');
            $Table->timestamp('called_in_at')->nullable()->comment('Zeitstempel des Call-Ins.');
            $Table->foreignId('check_in_entry_point_id')->nullable()->comment('ID des Zugangspunkts über den der Log-In erfolgte.')->constrained(table: 'visit_entry_points', indexName: 'visit_appointments_FK_check_in_entry_point_id');
            $Table->foreignId('check_out_entry_point_id')->nullable()->comment('ID des Zugangspunkts über den der Log-Out erfolgte.')->constrained(table: 'visit_entry_points', indexName: 'visit_appointments_FK_check_out_entry_point_id');
            $Table->timestamp('checked_in_at')->nullable()->comment('Zeitstempel des Check-Ins.');
            $Table->timestamp('checked_out_at')->nullable()->comment('Zeitstempel des Check-Outs.');
            $Table->foreignId('create_user_id')->constrained(table: 'users', indexName: 'visit_appointments_FK_create_user_id');
            $Table->boolean('drive_on_area_permitted')->default(0)->comment('Flag das anzeigt ob das Gelände befahren werden darf.');
            $Table->timestamp('entered_at')->nullable();
            $Table->string('ews_appointment_id', 500)->nullable()->comment('Exchange Appointment-ID');
            $Table->timestamp('finished_at')->nullable()->comment('Zeitstempel vom Abschluss des Besuchs.');
            $Table->boolean('internal_visit')->default(0);
            $Table->string('license_plate', 20)->nullable();
            $Table->string('license_plate_formatted', 25)->nullable();
            $Table->foreignId('loading_ramp_id')->nullable()->comment('Nur Logistik: Zugewiesene Laderampe.')->constrained(table: 'visit_locations', indexName: 'visit_appointments_FK_loading_ramp_id');
            $Table->foreignId('main_employee_id')->nullable()->comment('Bei regulären Besuchen die ID des verantwortlichen/begleitenden Mitarbeites. Bei Zufahrtsberechtigungen die ID des Mitarbeiters, für den die Berechtigung gilt.')->constrained(table: 'visit_employees', indexName: 'visit_appointments_FK_main_employee_id');
            $Table->foreignId('main_visitor_id')->nullable()->comment('ID des Besuchers bzw. des hauptverantwortlichen Besuchers bei Gruppenbesuchen.')->constrained(table: 'visit_visitors', indexName: 'visit_appointments_FK_main_visitor_id');
            $Table->string('order_number', 50)->nullable()->comment('Auftragsnummer.');
            $Table->foreignId('organizer_employee_id')->nullable()->comment('Mitarbeiter der sich um die Koordinierung des Termins kümmert.')->constrained(table: 'visit_employees', indexName: 'visit_appointments_FK_organizer_employee_id');
            /*
            **  Das Feld "process_status" war ursprünglich als Enum-Feld definiert. Da die Werte in der DB dann
            **  aber Strings sind, funktionierte der Vergleich auf größer oder kleiner nicht immer zuverlässig.  */
            $Table->tinyInteger('process_status')->default(ProcessStatus::Open->value)->comment('Siehe Enums/ProcessStatus: 0 = Open, 1 = Initialized, 2 = CheckedIn, 3 = CalledIn, 4 = Entered, 5 = Ongoing, 6 = CheckedOut, 7 = Finished');
            $Table->boolean('require_escort')->default(0)->comment('Flag das anzeigt ob das Begleitung erforderlich ist.');
            $Table->boolean('spontaneous_visit')->default(0)->comment('Flag das anzeigt, dass der Termin als Spontanbesuch erfasst wurde.');
            $Table->string('subject', 100)->nullable()->comment('Thema des Termins.');
            $Table->string('token', 7)->comment('Eindeutiges Token zur Identifizierung des Besuchs.');
            $Table->timestamp('valid_from')->comment('Zeitstempel für den geplanten Terminbeginn.');
            $Table->timestamp('valid_until')->comment('Zeitstempel für das voraussichtliche Terminende beziehungsweise Ende des Gültigkeitszeitraums (bei mehrtägigen/längerfristigen Terminen).');
            $Table->foreignId('visit_reason_id')->constrained(table: 'visit_reasons', indexName: 'visit_appointments_FK_visit_reason_id');
            $Table->json('visitors')->nullable()->comment('Daten zu den begleitenden Besuchern.');
            $Table->string('web_token', 33)->comment('Eindeutiges Token zur Verwendung in Links.');
            $Table->boolean('with_minors')->default(0)->comment('Flag das bei Gruppenbesuchen anzeigt ob Kinder Mitglieder der Gruppe sind.');

            $Table->index('bulk_appointment_id',      'visit_appointments_FKI_bulk_appointment_id');
            $Table->index('check_in_entry_point_id',  'visit_appointments_FKI_check_in_entry_point_id');
            $Table->index('check_out_entry_point_id', 'visit_appointments_FKI_check_out_entry_point_id');
            $Table->index('create_user_id',           'visit_appointments_FKI_create_user_id');
            $Table->index('loading_ramp_id',          'visit_appointments_FKI_loading_ramp_id');
            $Table->index('main_employee_id',         'visit_appointments_FKI_main_employee_id');
            $Table->index('main_visitor_id',          'visit_appointments_FKI_main_visitor_id');
            $Table->index('organizer_employee_id',    'visit_appointments_FKI_organizer_employee_id');
            $Table->index('visit_reason_id',          'visit_appointments_FKI_visit_reason_id');

            new CodeGenerator($Table, [
                'columns' => [
                    'uuid' => [
                        'guarded' => true,
                    ],
                    'active' => [
                        'sortable' => true,
                    ],
                    'bulk_appointment_id' => [
                        'reverseRelationName' => 'bulk-appointments',
                    ],
                    'bulk_informations' => [
                        'jsonType' => CodeGenerator::JSON_TYPE_OBJECT,
                    ],
                    'called_in_at' => [
                        'guarded' => true,
                        'sortable' => true,
                    ],
                    'check_in_entry_point_id' => [
                        'reverseRelationName' => 'appointment-check-ins',
                        'sortable' => true,
                    ],
                    'check_out_entry_point_id' => [
                        'reverseRelationName' => 'appointment-check-outs',
                        'sortable' => true,
                    ],
                    'checked_in_at' => [
                        'guarded' => true,
                        'sortable' => true,
                    ],
                    'checked_out_at' => [
                        'guarded' => true,
                        'sortable' => true,
                    ],
                    'create_user_id' => [
                        'guarded' => true,
                        'sortable' => true,
                    ],
                    'entered_at' => [
                        'guarded' => true,
                        'sortable' => true,
                    ],
                    'finished_at' => [
                        'guarded' => true,
                        'sortable' => true,
                    ],
                    'internal_visit' => [
                        'sortable' => true,
                    ],
                    'license_plate' => [
                        'guarded' => true,
                        'sortable' => true,
                    ],
                    'license_plate_formatted' => [
                        'sortable' => true,
                    ],
                    'loading_ramp_id' => [
                        'reverseRelationName' => 'appointments'
                    ],
                    'main_employee_id' => [
                        /*
                        **  Name der Relation aus der Sicht des Employees.
                        **  Dadurch, dass ein Mitarbeiter sowohl über die AppointmentEmployee-Tabelle als auch als verantwortlicher
                        **  Mitarbeiter mit einem Besuch verknüpft sein kann, gibt es beim Employee zwei automatische Relationen mit
                        **  dem Namen "appointments" (eine HasMany und eine HasManyThrough-Relation).
                        **  Mit dieser Option kann ein individueller Name vergeben werden, der beim Employee eingesetzt wird um die
                        **  Relation vom Hauptverantwortlichen Mitarbeiter zum Termin zu benennen. */
                        'reverseRelationName' => 'assigned-appointments',
                    ],
                    'main_visitor_id' => [
                        /*
                        **  Name der Relation aus der Sicht des Visitors.
                        **  Dadurch, dass ein Besucher sowohl über die AppointmentVisitor-Tabelle als auch als Hauptbesucher mit einem
                        **  Besuch verknüpft sein kann, gibt es beim Visitor zwei automatische Relationen mit dem Namen "appointments"
                        **  (eine HasMany und eine HasManyThrough-Relation).
                        **  Mit dieser Option kann ein individueller Name vergeben werden, der beim Visitor eingesetzt wird um die
                        **  Relation vom Hauptbesucher zum Termin zu benennen.
                        **
                        **  Die Tabelle AppointmentVisitor gibt es nicht mehr. Der individuelle Name der Relation wird trotzdem beibehalten. */
                        'reverseRelationName' => 'assigned-appointments',
                    ],
                    'organizer_employee_id' => [
                        /*
                        **  Name der Relation aus der Sicht des Employees.
                        **  Beschreibung wie oben bei "main_employee_id". */
                        'reverseRelationName' => 'organized-appointments',
                    ],
                    'process_status' => [
                        'guarded' => true,
                        'sortable' => true,
                    ],
                    'token' => [
                        'guarded' => true,
                        'readOnly' => true,
                    ],
                    'valid_from' => [
                        'cast' => 'datetime',
                        'sortable' => true,
                    ],
                    'valid_until' => [
                        'cast' => 'datetime',
                        'sortable' => true,
                    ],
                    'visit_reason_id' => [
                        'reverseRelationName' => 'appointments',
                    ],
                    'visitors' => [
                        'jsonType' => CodeGenerator::JSON_TYPE_ARRAY,
                    ],
                    'web_token' => [
                        'guarded' => true,
                        'readOnly' => true,
                    ],
                    'with_minors' => [
                        'sortable' => true,
                    ],
                ],
                'routing' => [
                    'terminal' => [
                        'actions' => [
                            'destroy',
                            'show',
                        ],
                        'method' => 'except',
                    ],
                ],
            ]);

        });
    } // up()


};
