<?php
/**
 * Import Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-modules/visitor-management
 * @subpackage  Imports
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanModules\VisitorManagement\Imports;


use BplanModules\VisitorManagement\Enums\ImportContext;
use BplanModules\VisitorManagement\Enums\ImportType;
use BplanModules\VisitorManagement\Services\ImportLogService;
use ErrorException;
use Maatwebsite\Excel\Concerns\Importable;
/*
**  Ein Trait der automatisch Events registriert, wenn die entsprechende Methode im Import implementiert
**  ist. Mögliche Events/Methoden sind:
**  - afterImport()
**  - afterSheet()
**  - beforeImport()
**  - beforeSheet()
*/
use Maatwebsite\Excel\Concerns\RegistersEventListeners;
/*
**  Verhindert die Verarbeitung von leeren Zeilen. */
use Maatwebsite\Excel\Concerns\SkipsEmptyRows;
/*
**  SkipsOnFailure sorgt dafür, dass Fehler, die bei der Validierung der Daten ermittelt werden, unterdrückt werden.
**  In Verbindung mit dem Trait SkipsFailures kann dann noch die Methode onFailure() implementiert werden, die für
**  jeden auftretenden Fehler aufgerufen wird. Alternativ kann auch die Methode SkipsFailures::failures() verwendet
**  werden um am Ende der Verarbeitung eine Liste aller Fehler zu erhalten.
**
**  WICHTIG!
**  Wenn onFailure() implementiert wird, dann muss die Funktionalität dieser Methode aus dem Trait mit in die eigene
**  Implementierung übernommen werden. Andernfalls liefert die Methode failures() ein leeres Ergebnis. */
use Maatwebsite\Excel\Concerns\SkipsFailures;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
/*
**  Ermöglicht die Registrierung (manuell) von Events mit der Methode registerEvents(), muss aber auch eingebunden werden,
**  wenn der Trait RegistersEventListeners (zur automaischen Registrierung von Events) verwendet werden soll.
**  */
use Maatwebsite\Excel\Concerns\WithEvents;
/*
**  Ermöglicht es, dass die Spaltenüberschriften aus der Importdatei als Schlüssel verwendet werden. */
use Maatwebsite\Excel\Concerns\WithHeadingRow;
/*
**  Progressbar in der Konsolenausgabe verwenden. */
use Maatwebsite\Excel\Concerns\WithProgressBar;
/*
**  Erkennt doppelte Zeilen in der Importdatei und ignoriert die Dubletten.
**  Es ist ausreichend diese use-Anweisung einzufügen. Weiterer Code ist nicht erforderlich, um diese
**  Funktionalität zu nutzen. */
use Maatwebsite\Excel\Concerns\WithSkipDuplicates;
use Maatwebsite\Excel\Concerns\WithStartRow;
use Maatwebsite\Excel\Events\AfterImport;
use Maatwebsite\Excel\Events\BeforeImport;
use Maatwebsite\Excel\Validators\Failure;


/**
 * Import Class
 *
 * @version     5.0.0 / 2025-01-29
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class BaseImport implements
    SkipsEmptyRows,
    SkipsOnFailure,
    WithEvents,
    WithHeadingRow,
    WithProgressBar,
    WithSkipDuplicates,
    WithStartRow
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    use Importable;
    use RegistersEventListeners;
    use SkipsFailures;


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Der Name der Backup-Datei
     * *
     * @var     string $_backupFileName
     */
    protected $_backupFileName;


    /**
     * Der vollständige Pfad zum Backup-Verzeichnis
     *
     * @var     string $_backupPath
     */
    protected $_backupPath;


    /**
     * @var   float $_begin
     */
    protected $_begin;


    /**
     * @var     bool $_devMode
     */
    protected $_devMode = false;


    /**
     * @var     array $_errors
     */
    protected $_errors = [];


    /**
     * @var     ImportContext $_Context
     */
    protected $_Context;


    /**
     * Die Daten der aktuellen Zeile als assoziativer Array
     *
     * Diese Variable wird in der Methode prepareForValidation() gefüllt. Wenn abgeleitete Klassen
     * diese Methode überschreiben, dann ist es erforderlich parent::prepareForValidation() aufzurufen
     * um diese Variable nutzen zu können.
     *
     * @var     array $_currentRowData
     */
    protected $_currentRowData;


    /**
     * Der Name der Importdatei
     *
     * @var     string $_importFileName
     */
    protected $_importFileName;


    /**
     * @var     ImportType $_ImportType
     */
    protected $_ImportType;


    /**
     * Der gesammelte Inhalt aller Failures, als Inhalt für die Log-Datei
     *
     * @var     string $_logFileContents
     */
    protected $_logFileContents = '';


    /**
     * @var     int $_logId
     */
    protected $_logId;


    /**
     * Der Name der Log-Datei
     *
     * @var     string $_logFileName
     */
    protected $_logFileName;


    /**
     * @var     bool $_moveMethod
     */
    protected $_moveMethod = 'rename';


    /**
     * @var     integer $_rowCountSaved
     */
    protected $_rowCountSaved = 0;


    /**
     * @var     integer $_rowCountTotal
     */
    protected $_rowCountTotal = 0;


    /**
     * Der vollständige Pfad zum Verzeichnis der Importdatei
     *
     * @var     string $_storagePath
     */
    protected $_storagePath;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Der Sub-Pfad, z.B. 'import/employees'
     *
     * @var     string $_subPath
     */
    protected static $_subPath;


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       ImportContext $Context
     *
     * @throws      ErrorException
     * *
     * @version     1.0.0 / 2024-09-24
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(ImportContext $Context)
    {
        $this->_Context = $Context;

        $this->_init();

    } // __construct()


    /**
     * Hängt Fehlermeldungen einer Zeile an den LogFile-Inhalt an
     *
     * @param       int $rowNumber
     *
     * @param       array $rowData
     *
     * @param       mixed $Failures
     *
     * @return 	    $this
     *
     * @version     1.2.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _appendLogFileContents(int $rowNumber, array $rowData, mixed $Failures): self
    {
        $logData = [
            'row' => $rowNumber,
            'errors' => [],
            'data' => $rowData,
        ];
        foreach ($Failures as $Failure) {
            $attribute = $Failure->attribute();

            $logData['errors'][$attribute] = $Failure->errors();
        }
        $this->_logFileContents .= json_encode($logData)
            ."\n";

        return $this;

    } // _appendLogFileContents()


    /**
     * Schließt den Import ab
     *
     * Diese Methode wird in der Methode afterImport() aufgerufen. Wenn afterImport() von einer abgeleiteten
     * Klasse überschrieben wird, dann ist es dort zwingend erforderlich am Ende der überschriebenen Methode
     * parent::afterImport() aufzurufen.
     *
     * @return 	    $this
     *
     * @version     1.2.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    final protected function _finish(): self
    {
        if (!file_exists($this->_backupPath)) {
            mkdir($this->_backupPath, 0777, true);
        }
        /*
        **  Importdatei ins Backup-Verzeichnis verschieben. */
        ($this->_moveMethod)($this->_storagePath.'/'.$this->_importFileName, $this->_backupPath.'/'.$this->_backupFileName);

        if ($this->_logFileContents !== '') {
            /*
            **  Fehlerbericht in die Log-Datei schreiben. */
            file_put_contents($this->_backupPath.'/'.$this->_logFileName, $this->_logFileContents);
        }
        $LogService = new ImportLogService();

        if ($this->_devMode === true) {
            dump([
                'errors' => $this->_errors,
                'failures' => $this->failures,
            ]);
        }
        $LogEntry = $LogService->create([
            'begin' => date('Y-m-d H:i:s', $this->_begin),
            'error_count' => count($this->_errors),
            'errors' => empty($this->_errors) ? null : $this->_errors,
            'failure_count' => count($this->failures),
            'failures' => empty($this->failures) ? null : $this->failures,
            'import_context' => $this->_Context->name,
            'import_file_name' => $this->_importFileName,
            'import_type' => $this->_ImportType->value,
            'row_count' => 0,
            'runtime' => microtime(true) - $this->_begin,
        ]);
        $this->_logId = $LogEntry->id;

        return $this;

    } // _finish()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-09-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _incrementRowCountSaved(): self
    {
        $this->_rowCountSaved++;

        return $this;

    } // _incrementRowCountSaved()


    /**
     *
     * @param       int $index
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-09-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _incrementRowCountTotal(int $index): self
    {
        static $lastCountedIndex = null;

        if ($lastCountedIndex === $index) {
            return $this;
        }
        $lastCountedIndex = $index;

        $this->_rowCountTotal++;

        return $this;

    } // _incrementRowCountTotal()


    /**
     * Initialisiert Variablen für den Import
     *
     * Diese Methode wird im Constructor aufgerufen. Wenn eine abgeleitete Klasse eine eigene init-Methode
     * implementiert, dann ist es zwingend erforderlich zu Beginn parent::_init() aufzurufen.
     *
     * @return      $this
     *
     * @version     1.0.0 / 2024-09-24
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    final protected function _init(): self
    {
        $this->_begin = microtime(true);

        if ($this->_devMode === true) {
            $this->_moveMethod = 'copy';
        }
        switch ($this->_Context) {
            case ImportContext::Queue:
            case ImportContext::Upload:
                $this->_backupPath = self::getBackupPath($this->_Context);
                $this->_storagePath = self::getStoragePath($this->_Context);
                break;

            default:
                throw new ErrorException('Invalid context "'.$this->_Context.'"');
        }
        return $this;

    } // _init()


    /**
     * Liefert einen Array mit den identifizierenden Feldern
     *
     * Die Standardmethode uniqueBy() kann wahlweise einen String oder einen Array zurückgeben. Diese Methode
     * nimmt den von uniqueBy() gelieferten Wert und bereitet ihn auf, so dass immer ein Array zur Verfügung
     * steht. Außderdem ist der Ergebnis-Array so aufgebaut, dass die Feldnamen sowohl als Schlüssel als auch
     * als Werte verwendet werden, so dass er problemlos mit array_intersect_key() verwendet werden kann.
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2024-09-24
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _uniqueBy(): array
    {
        if (!method_exists($this, 'uniqueBy')) {
            return [];
        }
        static $uniqueBy = null;

        if ($uniqueBy !== null) {
            return $uniqueBy;
        }
        $tmp = $this->uniqueBy();

        if (!is_array($tmp)) {
            $uniqueBy = [$tmp => $tmp];

        } else {
            $uniqueBy = array_combine($tmp, $tmp);
        }
        return $uniqueBy;

    } // _uniqueBy()


    /**
     * Führt Aktionen nach dem Import aus
     *
     * Wenn diese Methode von einer abgeleiteten Klasse überschrieben wird, dann ist es dort zwingend
     * erforderlich am Ende parent::afterImport() aufzurufen.
     *
     * @param       AfterImport $Event
     *
     * @version     1.0.0 / 2024-09-24
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function afterImport(AfterImport $Event): void
    {
        $this->_finish();

    } // afterImport()


    /**
     * Führt Aktionen vor dem Import aus
     *
     * @param       BeforeImport $Event
     *
     * @version     1.0.0 / 2024-09-24
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function beforeImport(BeforeImport $Event): void
    {
        //
    } // beforeImport()


    /**
     * Liefert einen Array mit Fehlern
     *
     * Bei den im Array gespeicherten Fehlern handelt es sich um Fehler die beim Speichern der
     * Datensätze aufgetreten sind.
     *
     * @return 	    array<array>
     *
     * @version     1.0.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getErrors(): array
    {
        return $this->_errors;

    } // getErrors()


    /**
     * Liefert einen Array mit Objekten vom Typ Maatwebsite\Excel\Validators\Failure
     *
     * Bei den Elementen im Array handelt es sich um Fehler die bei der Validierung der Daten
     * ermittelt wurden.
     *
     * @return 	    array<Failure>
     *
     * @version     1.0.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getFailures(): array
    {
        return $this->failures;

    } // getFailures()


    /**
     * Liefert die ID des Log-Eintrags des aktuellen Imports
     *
     * @return 	    int
     *
     * @version     1.0.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getLogId(): int
    {
        return $this->_logId;

    } // getLogId()


    /**
     *
     * @return 	    int
     *
     * @version     1.0.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getRowCountSaved(): int
    {
        return $this->_rowCountSaved;

    } // getRowCountSaved()


    /**
     *
     * @return 	    int
     *
     * @version     1.0.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getRowCountTotal(): int
    {
        return $this->_rowCountTotal;

    } // getRowCountTotal()


    /**
     * Definiert die Zeilennummer in der die Überschriften stehen
     *
     * @return      int
     *
     * @version     1.0.0 / 2024-09-24
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function headingRow(): int
    {
        return 1;

    } // headingRow()


    /**
     * Importiert die übergebene Datei
     *
     * Erwartet wird aktuell nur der vollständige Pfad zur Datei im Parameter $filePath.
     *
     * Die Verwendung der Parameter $disk und $readerType ist derzeit noch nicht implementiert. Es ist
     * jedoch vorgesehen, dass diese zu einem späteren Zeitpunkt noch implementiert werden, um die
     * Standardarbeitsweise der Facade-Methode import() nutzen zu können.
     *
     * todo         $disk und $readerType implementieren.
     *
     * @param       string $filePath
     *
     * @param       null|string $disk
     *
     * @param       null|string $readerType
     *
     * @version     1.0.0 / 2024-09-24
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function importFile(string $filePath, null|string $disk = null, null|string $readerType = null): void
    {
        if ($disk === null) {
            $importFileName = $this->setImportFileName($filePath);

            if (!file_exists($this->_storagePath)) {
                mkdir($this->_storagePath, 0777, true);
            }
            /*
            **  Importdatei ins Arbeitsverzeichnis verschieben, sofern sie nicht bereits dort liegt. */
            if ($filePath !== $this->_storagePath.'/'.$importFileName) {
                ($this->_moveMethod)($filePath, $this->_storagePath.'/'.$importFileName);
            }
            if ($this->_devMode === true) {
                dump($this->_storagePath.'/'.$importFileName);
            }
            $this->import($this->_storagePath.'/'.$importFileName, $disk, $readerType);
        }
    } // importFile()


    /**
     * In dieser Methode kann Code definiert werden, der prüft ob die übergebene Zeile leer ist
     *
     * @param       array $row
     *
     * @return      bool
     *
     * @version     1.0.0 / 2024-09-24
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function isEmptyWhen(array $row): bool
    {
        $firstKey = array_key_first($row);
        /*
        **  Prüfen ob es sich bei dem ersten Zeichen des ersten Feldes um eine Raute (#) handelt.
        **  Zeilen, die mit einem Rauten-Symbol beginnen, werden als Kommentarzeilen angesehen. */
        return isset($row[$firstKey][0]) && $row[$firstKey][0] === '#';

    } // isEmptyWhen()


    /**
     * Diese Methode wird für jede fehlerhafte Zeile aufgerufen
     *
     * @param       Failure ...$Failures
     *
     * @requires    Maatwebsite\Excel\Concerns\SkipsOnFailure
     * @requires    Maatwebsite\Excel\Concerns\WithValidation
     *              Damit Felder mit der rules()-Methode validiert werden und im Fehlerfall die Methode aufgerufen wird.
     *              Alternativ kann auch noch der Trait Maatwebsite\Excel\Concerns\SkipsFailures eingebunden werden,
     *              der die Methode failures() zur Verfügung stellt, die eine Collection mit allen Fehlern liefert.
     *
     * @version     1.0.0 / 2024-09-24
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function onFailure(Failure ...$Failures): void
    {
        /*
        **  Funktionalität aus der Methode SkipFailures::onFailure() übernommen.
        **
        **  Ohne diese Codezeile würde SkipFailures::failures() am Ende ein leeres Ergebnis liefern. Wenn onFailure()
        **  nicht überschrieben wird, dann funktioniert SkipFailures::failures() wie erwartet. */
        $this->failures = array_merge($this->failures, $Failures);
        /*
        **  Ersten Fehler aus der Liste ermitteln.
        **  Die Zeilennummer und die Daten der Zeile werden von diesem Fehler verwendet. Sie werden nur einmalig
        **  benötigt. */
        $CurrentFailure = current($Failures);

        $this->_appendLogFileContents($CurrentFailure->row(), $CurrentFailure->values(), $Failures);

    } // onFailure()


    /**
     * Bereitet Werte einer Zeile vor der Validierung auf
     *
     * @param       array $data
     *              Die aktuelle Zeile als Array.
     *
     * @param       int $index
     *              Die Zeilennummer.
     *
     * @return      array
     *
     * @requires    Maatwebsite\Excel\Concerns\WithValidation
     *              Diese Methode wird automatisch in der Methode Maatwebsite\Excel\Row::toArray() aufgerufen. Das führt
     *              leider dazu, dass der Code jedes Mal ausgeführt wird, wenn toArray() verwendet wird. Da dabei aber
     *              immer das unveränderte Row-Objekt verwendet wird, kommt es glücklicherweise nicht dazu, dass Daten
     *              doppelt modifiziert werden.
     *
     * @version     1.1.0 / 2024-09-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function prepareForValidation($data, $index): array
    {
        $this->_currentRowData = $data;

        $this->_incrementRowCountTotal($index);

        return $data;

    } // prepareForValidation()


    /**
     * Ermittelt den Dateinamen aus dem übergebenen Dateipfad und setzt diverse Objektvariablen
     *
     * @param       string $filePath
     *              Der vollständige Pfad der Datei.
     *
     * @return      string Gibt den extrahierten Dateinamen zurück.
     *
     * @version     1.0.0 / 2024-09-24
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function setImportFileName(string $filePath): string
    {
        $importFileName = pathinfo($filePath, PATHINFO_BASENAME);

        $this->_importFileName = $importFileName;

        $this->_backupFileName = date('Ymd.His').'.'.$this->_importFileName;
        $this->_logFileName = $this->_backupFileName.'.log';

        return $importFileName;

    } // setImportFileName()


    /**
     * Definiert die Zeilenummer bei der der Import beginnen soll
     *
     * @return      int
     *
     * @version     1.0.0 / 2024-09-24
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function startRow(): int
    {
        return 2;

    } // startRow()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       ImportContext $Context
     *
     * @throws      ErrorException
     *
     * @return      string
     *
     * @version     1.0.1 / 2024-11-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getBackupPath(ImportContext $Context): string
    {
        $storagePath = str_replace('\\', '/', storage_path('app'));

        switch ($Context) {
            case ImportContext::Queue:
            case ImportContext::Upload:
                return $storagePath.'/backup/'.$Context->value.'/'.static::$_subPath;
                break;

            default:
                throw new ErrorException('Invalid context "'.$Context.'"');
        }
    } // getBackupPath()


    /**
     *
     * @param       ImportContext $Context
     *
     * @throws      ErrorException
     *
     * @return      string
     *
     * @version     1.0.0 / 2024-09-24
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getStoragePath(ImportContext $Context): string
    {
        $storagePath = str_replace('\\', '/', storage_path('app'));

        switch ($Context) {
            case ImportContext::Queue:
                return $storagePath.'/'.ImportContext::Queue->value.'/'.static::$_subPath;
                break;

            case ImportContext::Upload:
                return $storagePath.'/tmp/'.ImportContext::Upload->value.'/'.static::$_subPath;
                break;

            default:
                throw new ErrorException('Invalid context "'.$Context.'"');
        }
    } // getStoragePath()


} // class BaseImport implements ... {}
