<?php
/**
 * Repository Class
 *
 * Wegen individueller Anpassungen aus der automatischen Generierung genommen.
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @generated   2025-04-13 18:30:43
 * @package     bplan-modules/visitor-management
 * @subpackage  Repositories
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanModules\VisitorManagement\Repositories;


use BplanComponents\LaravelVegaRpc\Enums\CheckResultCode;
use BplanModules\VisitorManagement\Enums\AttachmentFileType;
use BplanModules\VisitorManagement\Enums\EntryPointDirection;
use BplanModules\VisitorManagement\Enums\ProcessStatus;
use BplanModules\VisitorManagement\Enums\VisitTypeIdentifier;
use BplanModules\VisitorManagement\Models\VisitAppointment;
use BplanModules\VisitorManagement\Models\VisitType;
use BplanModules\VisitorManagement\Services\VisitAppointmentService;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;


/**
 * Repository Class
 *
 * @version     2.1.0 / 2025-04-13
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class VisitAppointmentRepository
{

/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Die Dateinamen der Standard-Dateianhänge
     *
     * ACHTUNG
     * Dem Platzhalter "{lang}" ist hier bewusst kein Punkt (.) vorangestellt. Dieser wird bei der Ersetzung des Platzhalters
     * eingesetzt.
     *
     * @var         array $_attachmentFileNames
     * @version     1.0.0 / 2024-11-11
     */
    private static $_attachmentFileNames = [
        // AttachmentFileType::HouseRules->name => AttachmentFileType::HouseRules->value.'{lang}.pdf',
        AttachmentFileType::RouteDescription->name => AttachmentFileType::RouteDescription->value.'{lang}.pdf',
    ];


    /**
     * @var     string $_defaultLanguage
     */
    private static $_defaultLanguage;


    /**
     * Die Dateinamen der Sicherheitsunterweisungen
     *
     * Die Reihenfolge der Elemente im Array definiert die Reihenfolge der Suche (vom Speziellen zum Allgemeinen ! NICHT ÄNDERN !).
     *
     * ACHTUNG
     * Dem Platzhalter "{lang}" ist hier bewusst kein Punkt (.) vorangestellt. Dieser wird bei der Ersetzung des Platzhalters
     * eingesetzt.
     *
     * @var         array $_safetyBriefingFileNames
     * @version     1.0.0 / 2024-11-11
     */
    private static $_safetyBriefingFileNames = [
        AttachmentFileType::SafetyBriefing->value.'.{visit-type-group}.{visit-type-identifier}{lang}.pdf',
        AttachmentFileType::SafetyBriefing->value.'.{visit-type-group}{lang}.pdf',
        AttachmentFileType::SafetyBriefing->value.'.general{lang}.pdf',
    ];


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       string $licensePlate
     *
     * @param       null & $Appointment
     *
     * @return 	    void
     *
     * @version     1.2.0 / 2024-12-10
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _checkEntryPermission(string $licensePlate, null & $Appointment): bool
    {
dd(7);
        /*
        **  Ein Fahrzeug finden dessen Termin zum aktuellen Zeitpunkt passt.
        **  Dabei muss der Terminbeginn in der Vergangenheit liegen und das Terminende in
        **  der Zukunft liegen. */
        $Query = VisitAppointment::where('license_plate', '=', $licensePlate)
            ->where('active', '=', 1)
            ->where('process_status', '>=', ProcessStatus::CalledIn->value)
            ->where('process_status', '<', ProcessStatus::Finished->value)
            ->where('valid_from', '<=', now())
            ->where('valid_until', '>=', now());

        $Appointment = $Query->first();

        if ($Appointment === null) {
            throw new Exception('Appointment not found or out of date', CheckResultCode::NotFound->value);
        }
        /*
            Fehlermeldung noch verfeinern.
        */
        return true;

    } // _checkEntryPermission()


    /**
     *
     * @param       string $licensePlate
     *
     * @param       null & $Appointment
     *
     * @return 	    bool
     *
     * @version     1.2.0 / 2024-12-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _checkExitPermission(string $licensePlate, null & $Appointment): bool
    {
        /*
        **  Ein Fahrzeug finden dessen Termin zum aktuellen Zeitpunkt passt.
        **  Der Termin darf nicht finalisiert sein und der Terminbeginn muss in der Vergangenheit
        **  liegen. */
        $Query = VisitAppointment::where('license_plate', '=', $licensePlate)
            ->where('process_status', '<', ProcessStatus::Finished->value)
            ->where('valid_from', '<', now());

        $Appointment = $Query->first();

        if ($Appointment === null) {
            throw new Exception('Appointment not found', CheckResultCode::NotFound->value);
        }
        if (VisitAppointmentRepository::hasEquipmentItems($Appointment) === true) {
            throw new Exception('Check out is not permitted due to issued equipment', CheckResultCode::Forbidden->value);
        }
        return true;

    } // _checkExitPermission()


    /**
     * Ermittelt einen Mail-Anhang
     *
     * Es wird immer erst versucht einen Anhang in der bevorzugten Sprache des Besuchers zu ermitteln. Wenn der nicht
     * gefunden werden kann, dann wird nach einem Anhang in der Standardsprache gesucht. Kann auch dieser Anhang nicht
     * gefunden werden, wird eine Exception geworfen.
     *
     * @param       AttachmentFileType $AttachmentFileType
     *
     * @param       string $preferredLanguage
     *
     * @throws      Exception
     *
     * @return 	    string
     *
     * @version     1.1.0 / 2024-12-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _determineAttachmentFile(AttachmentFileType $AttachmentFileType, string $preferredLanguage): string
    {
        $fileName = self::$_attachmentFileNames[$AttachmentFileType->name];

        $attachmentFile = null;

        $Disk = Storage::disk('attachments');

        foreach (self::_prepareLanguages($preferredLanguage) as $key => $language) {
            $which = ($key === 0) ? 'preferred' : 'default';

            $realFileName = str_replace('{lang}', '.'.$language, $fileName);

            if ($Disk->exists($realFileName)) {
                $attachmentFile = $Disk->path($realFileName);

                break;
            }
            Log::notice('Missing attachment file "'.$realFileName.'" in '.$which.' language ('.$language.') in '.__METHOD__.'().');
        }
        if ($attachmentFile === null) {
            $errorFileName = str_replace('{lang}', '', $fileName);

            Log::error('Missing attachment file "'.$errorFileName.'" in '.__METHOD__.'().');

            throw new \Exception('Missing mail attachment "'.$errorFileName.'".');
        }
        return $attachmentFile;

    } // _determineAttachmentFile()


    /**
     * Ermittelt die Sicherheitsunterweisung als Mail-Anhang
     *
     * Hier werden Dateien gesucht die entweder zur Besuchsgruppe UND zum Besuchstyp oder NUR zum Besuchstyp passen oder
     * die allgemein sind (in dieser Reihenfolge). In jedem Fall wird immer zuerst nach einer Datei in der bevorzugten
     * Sprache des Besuchers und anschließend in der Standardsprache gesucht.
     * Daraus ergibt sich, dass die generelle Sicherheitsunterweisung in der Standardsprache in jedem Fall immer
     * vorhanden sein muss.
     *
     * @param       VisitType $VisitType
     *
     * @param       string $preferredLanguage
     *
     * @throws      Exception
     *
     * @return 	    string
     *
     * @version     1.1.1 / 2024-12-17
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _determineSafetyBriefingFile(string $visitTypeGroup, string $visitTypeIdentifier, string $preferredLanguage): string
    {
        $safetyBriefingFile = null;

        $Disk = Storage::disk('attachments');

        foreach (self::$_safetyBriefingFileNames as $fileNameTemplate) {
            foreach (self::_prepareLanguages($preferredLanguage) as $key => $language) {
                $which = ($key === 0) ? 'preferred' : 'default';

                $fileName = str_replace('{lang}', '.'.$language, $fileNameTemplate);
                $fileName = str_replace('{visit-type-group}', $visitTypeGroup, $fileName);

                $realFileName = str_replace('{visit-type-identifier}', $visitTypeIdentifier, $fileName);

                if ($Disk->exists($realFileName)) {
                    $safetyBriefingFile = $Disk->path($realFileName);

                    break 2;
                }
                Log::notice('Missing safety briefing for "'.$realFileName.'" in '.$which.' language ('.$preferredLanguage.') in '.__METHOD__.'().');
            }
        }
        if ($safetyBriefingFile === null) {
            Log::error('Missing safety briefing "'.$visitTypeGroup.'/'.$visitTypeIdentifier.'" in default language ('.self::$_defaultLanguage.') in '.__METHOD__.'().');

            throw new \Exception('Missing safety briefing for "'.$visitTypeGroup.'/'.$visitTypeIdentifier.'".');
        }
        return $safetyBriefingFile;

    } // _determineSafetyBriefingFile()


    /**
     *
     * @param       string $preferredLanguage
     *
     * @return 	    array
     *
     * @version     1.3.0 / 2025-01-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _prepareLanguages(string $preferredLanguage): array
    {
        static $cache = [];

        $cacheKey = md5(serialize(func_get_args()));

        if (isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        self::$_defaultLanguage = config('app.fallback_locale', 'en');
        /*
        **  Sprachen zusammenstellen.
        **  Unabhängig von der tatsächlich verfügbaren Sprachen, werden hier lediglich die bevorzugte Sprache des Besuchers
        **  und die Standardsprache des Systems abgearbeitet. Der zusammengestellte Array wird noch bereinigt, für den
        **  Fall dass beide Sprachen identisch sind. */
        return $cache[$cacheKey] = array_unique([
            $preferredLanguage,
            self::$_defaultLanguage,
        ]);
    } // _prepareLanguages()


    /**
     *
     * @param       string $licensePlate
     *              Das zu überprüfende Kennzeichen.
     *
     * @param       array $requestParams
     *              Alle Daten aus dem Request.
     *
     * @param       array|string & $result
     *              Falls der Parameter $result übergeben wurde, wird er mit einem Text gefüllt,
     *              der Details zum Ergebnis beinhaltet.
     *
     * @return      bool
     *
     * @version     2.2.0 / 2024-11-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function checkPermission(string $licensePlate, array $requestParams, array|null|string & $result = null): bool
    {
        $entryPointToken = request()->query('entry-point');

        if ($entryPointToken === null) {
            throw new Exception('Missing required Parameter [entry-point]', CheckResultCode::BadRequest->value);
        }
        $EntryPoint = VisitEntryPointRepository::getModelByToken($entryPointToken);

        if ($EntryPoint === null) {
            throw new Exception('Unknwon entry point ['.$entryPointToken.']', CheckResultCode::BadRequest->value);
        }
        $error = null;
        $permitted = false;
        $Appointment = null;
        /*
            @todo   Wenn Direction BOTH ist, dann könnte die aktuelle Richtung mit einem Query-Parameter
                    übergeben werden.
        */
        try {
            if ($EntryPoint->direction === EntryPointDirection::In->name) {
                $permitted = self::_checkEntryPermission($licensePlate, $Appointment);
            } else {
                $permitted = self::_checkExitPermission($licensePlate, $Appointment);
            }
        } catch (Exception $E) {
            throw $E;
        }
        if ($permitted === false) {
            $result = __('vega-rpc::vega-rpc.check-result.message.Forbidden', ['plate' => $licensePlate]);

            return false;
        }
        $result = __('vega-rpc::vega-rpc.check-result.message.OK', ['plate' => $licensePlate]);

        $AppointmentService = new VisitAppointmentService();

        if ($EntryPoint->direction === EntryPointDirection::In->name) {
            $AppointmentService->setProcessStatusEntered($Appointment, [
                'check_in_entry_point_id' => $EntryPoint->id ?? null,
            ]);
        } else {
            $Appointment = $AppointmentService->setProcessStatusCheckedOut($Appointment, [
                'check_out_entry_point_id' => $EntryPoint->id ?? null,
            ]);
        }
        return true;

    } // checkPermission()


    /**
     *
     * @return 	    string
     *
     * @version     2.0.0 / 2024-12-17
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function determineAttachmentFile(VisitAppointment|int|string $appointment, AttachmentFileType $AttachmentFileType, null|string $lang = null): string
    {
        static $cache = [];

        $cacheKey = md5(serialize(func_get_args()));

        if (isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        if (!$appointment instanceof VisitAppointment) {
            $appointment = self::getById($appointment);
        }
        $Visitor = VisitVisitorRepository::getById($appointment->main_visitor_id);

        $visitTypeGroup = 'none';
        $visitTypeIdentifier = 'none';

        if ($appointment->visit_reason_id !== null) {
            $VisitType = VisitTypeRepository::getByVisitReason($appointment->visit_reason_id);

            $visitTypeGroup = $VisitType->visit_group;
            $visitTypeIdentifier = $VisitType->identifier;
        }
        if ($AttachmentFileType === AttachmentFileType::SafetyBriefing) {
            return $cache[$cacheKey] = self::_determineSafetyBriefingFile($visitTypeGroup, $visitTypeIdentifier, $Visitor->preferred_language);
        }
        if ($lang === null) {
            $lang = $Visitor->preferred_language;
        }
        return $cache[$cacheKey] = self::_determineAttachmentFile($AttachmentFileType, $lang);

    } // determineAttachmentFile()


    /**
     * Returns a collection of models
     *
     * @param       bool $ignoreCache
     *              Controls the use of the method cache when determining the result.
     *              If TRUE is passed, a database access is carried out even if the method cache
     *              already contains a suitable entry. The storage of a value in the method cache
     *              is not affected by this parameter.
     *
     * @param       bool $withInactive
     *              By default, only the active data records are returned. By setting this parameter
     *              to TRUE, the inactive data records can also be read out.
     *
     * @return      Collection
     *
     * @version     1.0.0 / 2025-04-13
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getAll(bool $ignoreCache = false, bool $withInactive = false): Collection
    {
        static $cache = [];

        $cacheKey = md5(serialize(func_get_args()));

        if ($ignoreCache === false && isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        $Query = VisitAppointment::query();

        if ($withInactive === false) {
            $Query->where('active', '=', 1);
        }
        return $cache[$cacheKey] = $Query->get();

    } // getAll()


    /**
     * Liefert alle Einzeltermine zu einem Bulk-Appointment
     *
     * @param       int|string $bulkAppointmentId
     *
     * @return      Collection
     *
     * @version     1.0.0 / 2025-01-15
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getBulkAppointments(int|string $bulkAppointmentId): Collection
    {
        static $cache = [];

        $cacheKey = md5(serialize(func_get_args()));

        if (isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        return $cache[$cacheKey] = VisitAppointment::where('bulk_appointment_id', '=', $bulkAppointmentId)
            ->get();

    } // getBulkAppointments()


    /**
     * Returns the model instance for the given ID
     *
     * @param       int|string $id
     *
     * @param       bool $ignoreCache
     *              Controls the use of the method cache when determining the result.
     *              If TRUE is passed, a database access is carried out even if the method cache
     *              already contains a suitable entry. The storage of a value in the method cache
     *              is not affected by this parameter.
     *
     * @param       bool $returnOrFail
     *              Controls the return of the method.
     *              If no suitable entry can be found in the database, either NULL is returned (TRUE)
     *              or an exception is thrown (FALSE). This allows the result of the method call to
     *              be used directly in the Api controller.
     *
     * @throws      Illuminate\Database\Eloquent\ModelNotFoundException
     *
     * @return      VisitAppointment
     *
     * @version     2.0.0 / 2025-04-13
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getById(int|string $id, bool $ignoreCache = false, bool $returnOrFail = true): VisitAppointment
    {
        static $cache = [];

        $cacheKey = md5(serialize(func_get_args()));

        if ($ignoreCache === false && isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        $Query = VisitAppointment::query();

        if ($returnOrFail === true) {
            return $Query->find($id);
        }
        return $Query->findOrFail($id);

    } // getById()


    /**
     * Liefert die Zufahrtsberechtigungen eines Mitarbeiters
     *
     * @param       int|string $employee_id
     *
     * @param       bool $ignoreCache
     *              Steuert die Verwendung des Methoden-Cache beim Ermitteln des Ergebnisses.
     *              Wenn TRUE übergeben wird, dann wird ein Datenbankzugriff durchgeführt, auch wenn
     *              der Methoden-Cache bereits einen passenden Eintrag enthält.
     *              Die Speicherung eines Wertes im Methoden-Cache wird von diesem Parameter nicht
     *              beeinflusst.
     *
     * @return      Collection
     *
     * @version		1.2.0 / 2024-11-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getEmployeeAccessAuthorizations(int|string $employee_id, bool $ignoreCache = false): Collection
    {
        static $cache = [];

        $cacheKey = md5(serialize(func_get_args()));

        if ($ignoreCache === false && isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        $VisitReasons = VisitReasonRepository::getByVisitType(VisitTypeIdentifier::AccessAuthorization);

        $Query = VisitAppointment::where('main_employee_id', '=', $employee_id)
            ->whereIn('visit_reason_id', $VisitReasons->pluck('id'));

        return $cache[$cacheKey] = $Query->get();

    } // getEmployeeAccessAuthorizations()


    /**
     * Liefert das Appointment-Model zur ID
     *
     * @param       int|string $id
     *
     * @param       bool $ignoreCache
     *              Steuert die Verwendung des Methoden-Cache beim Ermitteln des Ergebnisses.
     *              Wenn TRUE übergeben wird, dann wird ein Datenbankzugriff durchgeführt, auch wenn
     *              der Methoden-Cache bereits einen passenden Eintrag enthält.
     *              Die Speicherung eines Wertes im Methoden-Cache wird von diesem Parameter nicht
     *              beeinflusst.
     *
     * @param       bool $ignoreFinishedStatus
     *              Standardmäßig werden nur Appointments ausgelesen die noch nicht abgeschlossen
     *              sind. Mit diesem Parameter lässt sich steuern, dass der finished-Status ignoriert
     *              wird.
     *
     * @throws      ModelNotFoundException Wirft eine ModelNotFoundException, wenn kein Appointment
     *              zum Token gefunden werden konnte.
     *
     * @return      VisitAppointment
     *
     * @version     2.3.0 / 2024-12-09
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getModelByID(int|string $id, bool $ignoreCache = false, bool $ignoreFinishedStatus = false): VisitAppointment
    {
        static $cache = [];

        $cacheKey = md5(serialize(func_get_args()));

        if ($ignoreCache === false && isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        $Query = VisitAppointment::query();

        if ($ignoreFinishedStatus === false) {
            $Query->where('process_status', '<', ProcessStatus::Finished->value);
        }
        $Appointment = $Query->findOrFail($id);

        return $cache[$cacheKey] = $Appointment;

    } // getModelByID()


    /**
     * Liefert das Appointment-Model zur Auftragsnummer
     *
     * @param       string $orderNumber
     *
     * @param       bool $ignoreCache
     *              Steuert die Verwendung des Methoden-Cache beim Ermitteln des Ergebnisses.
     *              Wenn TRUE übergeben wird, dann wird ein Datenbankzugriff durchgeführt, auch wenn
     *              der Methoden-Cache bereits einen passenden Eintrag enthält.
     *              Die Speicherung eines Wertes im Methoden-Cache wird von diesem Parameter nicht
     *              beeinflusst.
     *
     * @param       bool $ignoreFinishedStatus
     *              Standardmäßig werden nur Appointments ausgelesen die noch nicht abgeschlossen
     *              sind. Mit diesem Parameter lässt sich steuern, dass der finished-Status ignoriert
     *              wird.
     *
     * @throws      ModelNotFoundException Wirft eine ModelNotFoundException, wenn kein Appointment
     *              zum Token gefunden werden konnte.
     *
     * @return      VisitAppointment
     *
     * @version     1.3.0 / 2025-01-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getModelByOrderNumber(string $orderNumber, bool $ignoreCache = false, bool $ignoreFinishedStatus = false): VisitAppointment|null
    {
        static $cache = [];

        $cacheKey = md5(serialize(func_get_args()));

        if ($ignoreCache === false && isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        /*
        **  Es werden nur die Appointments durchsucht, deren Beginn vor dem Ende des aktuellen Tages
        **  liegt. So wird sichergestellt, dass kein vorzeitiger CheckIn am Terminal möglich ist. */
        $Query = VisitAppointment::where('order_number', '=', $orderNumber)
            ->where('valid_from', '<=', Carbon::now()->endOfDay());

        if ($ignoreFinishedStatus === false) {
            $Query->where('process_status', '<', ProcessStatus::Finished->value);
        }
        $Appointment = $Query->firstOrFail();

        return $cache[$cacheKey] = $Appointment;

    } // getModelByOrderNumber()


    /**
     * Liefert das Appointment-Model zum Token
     *
     * @param       string $token
     *
     * @param       bool $ignoreCache
     *              Steuert die Verwendung des Methoden-Cache beim Ermitteln des Ergebnisses.
     *              Wenn TRUE übergeben wird, dann wird ein Datenbankzugriff durchgeführt, auch wenn
     *              der Methoden-Cache bereits einen passenden Eintrag enthält.
     *              Die Speicherung eines Wertes im Methoden-Cache wird von diesem Parameter nicht
     *              beeinflusst.
     *
     * @param       bool $ignoreFinishedStatus
     *              Standardmäßig werden nur Appointments ausgelesen die noch nicht abgeschlossen
     *              sind. Mit diesem Parameter lässt sich steuern, dass der finished-Status ignoriert
     *              wird.
     *
     * @param       bool $ignoreValidTime
     *              Um einen vorzeitigen CheckIn am Terminal zu unterbinden, wird ein Appointment nur dann
     *              berücksichtigt, wenn der Terminbeginn vor dem Ende des aktuellen Tages liegt. Um diese
     *              Bedingung zu deaktiveren, kann hier TRUE übergeben werden
     *
     * @throws      ModelNotFoundException Wirft eine ModelNotFoundException, wenn kein Appointment
     *              zum Token gefunden werden konnte.
     *
     * @return      VisitAppointment
     *
     * @version     3.0.0 / 2025-01-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getModelByToken(string $token, bool $ignoreCache = false, bool $ignoreFinishedStatus = false, bool $ignoreValidTime = false): VisitAppointment|null
    {
        static $cache = [];

        $cacheKey = md5(serialize(func_get_args()));

        if ($ignoreCache === false && isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        if ($token[0] === VisitAppointment::WEB_TOKEN_PREFIX) {
            $Query = VisitAppointment::where('web_token', '=', $token);
        } else {
            $Query = VisitAppointment::where('token', '=', $token);
        }
        if ($ignoreValidTime !== true) {
            /*
            **  Es werden nur die Appointments durchsucht, deren Beginn vor dem Ende des aktuellen Tages
            **  liegt. So wird sichergestellt, dass kein vorzeitiger CheckIn am Terminal möglich ist. */
            $Query->where('valid_from', '<=', Carbon::now()->endOfDay());
        }
        if ($ignoreFinishedStatus !== true) {
            $Query->where('process_status', '<', ProcessStatus::Finished->value);
        }
        $Appointment = $Query->firstOrFail();

        return $cache[$cacheKey] = $Appointment;

    } // getModelByToken()


    /**
     * Liefert alle Visitor-IDs zu Terminen
     *
     * @param       VisitAppointment|int|null|string $appointment
     *
     * @param       Builder|null $Query
     *              Ein Builder-Objekt mit vordefinierten Bedingungen zur Beschränkung
     *              des Ergebnisses.
     *
     * @return      array<int, int>
     *
     * @version     1.0.0 / 2025-03-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getVisitorIds(VisitAppointment|int|null|string $appointment = null, Builder|null $Query = null): array
    {
        if ($Query === null) {
            $Query = VisitAppointment::query();
        }
        if ($appointment !== null) {
            if ($appointment instanceof VisitAppointment) {
                $Query->where('id', '=', $appointment->id);

            } else {
                $Query->where('id', '=', $appointment);
            }
        }
        $Appointments = $Query->get();
        /*
        **  IDs der Gruppenbesucher ermitteln. */
        $additionalVisitorIds = $Appointments->whereNotNull('visitors')->flatMap(function ($Appointment) {
            return collect($Appointment->visitors)->pluck('id');

        })->map(function ($item) {
            return (int) $item;

        })->toArray();
        /*
        **  IDs der Hauptbesucher ermitteln. */
        $mainVisitorIds = $Appointments->whereNotNull('main_visitor_id')
            ->pluck('main_visitor_id')
            ->toArray();
        /*
        **  IDs zusammenführen und eine bereinigte Liste der IDs zurückgeben. */
        $allIds = array_merge($additionalVisitorIds, $mainVisitorIds);

        return array_unique($allIds);

    } // getVisitorIds()


    /**
     *
     * @param       int|string|Appointment $appointment
     *
     * @return      bool
     *
     * @version     1.0.0 / 2024-12-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function hasEquipmentItems(int|string|VisitAppointment $appointment): bool
    {
        return VisitAppointmentEquipmentItemRepository::getByAppointment($appointment)->isNotEmpty();

    } // hasEquipmentItems()


} // class VisitAppointmentRepository {}
