<?php
/**
 * Import Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-modules/visitor-management
 * @subpackage  Imports
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanModules\VisitorManagement\Imports;


use App\Models\User;
use App\Services\UserService;
use BplanBase\Globals\Helpers\StringHelper;
use BplanBase\Globals\Models\Role;
use BplanBase\Globals\Rules\RequiredIfEmpty;
use BplanBase\Globals\Services\RoleUserService;
use BplanModules\VisitorManagement\Enums\ImportType;
use BplanModules\VisitorManagement\Enums\LanguageContext;
use BplanModules\VisitorManagement\Models\VisitEmployee;
use BplanModules\VisitorManagement\Models\VisitContactGroup;
use BplanModules\VisitorManagement\Models\VisitLanguage;
use BplanModules\VisitorManagement\Services\VisitContactGroupService;
use Maatwebsite\Excel\Concerns\Importable;
use Maatwebsite\Excel\Concerns\OnEachRow;
use Maatwebsite\Excel\Concerns\WithUpsertColumns;
use Maatwebsite\Excel\Concerns\WithUpserts;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Events\AfterImport;
use Maatwebsite\Excel\Events\BeforeImport;
use Maatwebsite\Excel\Row;
use Maatwebsite\Excel\Validators\Failure;


/**
 * Import Class
 *
 * @version     7.6.0 / 2025-04-29
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class EmployeesImport extends BaseImport implements
    OnEachRow,
    WithUpsertColumns,
    WithUpserts,
    WithValidation
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    use Importable;


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Die Liste der ContactGroups, indiziert über ihr Label
     *
     * @var     array $_contactGroups
     */
    protected $_contactGroups;


    /**
     * Ein ContactGroupService-Objekt, mit dem bei Bedarf neue ContactGroups angelegt werden
     *
     * @var     ContactGroupService $_ContactGroupService
     */
    protected $_ContactGroupService;


    /**
     * @var     ImportType $_ImportType
     */
    protected $_ImportType = ImportType::Employee;


    /**
     * Die Liste der Roles, indiziert über ihren Identifier
     *
     * @var     array $_roles
     */
    protected $_roles;


    /**
     * Ein RoleUserService-Objekt, mit dem bei Bedarf neue RoleUser angelegt werden
     *
     * @var     RoleUserService $_RoleUserService
     */
    protected $_RoleUserService;


    /**
     * Die Liste der Users, indiziert über ihre E-Mail-Adresse
     *
     * @var     array $_users
     */
    protected $_users;


    /**
     * Ein UserService-Objekt, mit dem bei Bedarf neue User angelegt werden
     *
     * @var     UserService $_UserService
     */
    protected $_UserService;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string $_subPath
     */
    protected static $_subPath = 'import/employees';


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Ermittelt die ID zum übergebenen ContactGroup-Namen
     *
     * @param       string $contactGroup
     *
     * @return      int
     *
     * @version     1.4.0 / 2025-01-06
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _determineContactGroupId(null|string $contactGroup): int|null
    {
        if ($contactGroup === null) {
            return null;
        }
        if (!isset($this->_contactGroups[$contactGroup])) {
            $LanguageCollection = VisitLanguage::where('context', '!=', LanguageContext::None->value)
                ->get();

            $labels = [];

            foreach ($LanguageCollection as $Language) {
                $labels[$Language->code] = $contactGroup;
            }
            /*
            **  Wenn es noch keine passende ContactGroup gibt, dann wird eine neue Gruppe angelegt. */
            $ContactGroup = $this->_ContactGroupService->create([
                'labels' => (object) $labels
            ]);
            $this->_contactGroups[$contactGroup] = $ContactGroup->toArray();
        }
        return $this->_contactGroups[$contactGroup]['id'];

    } // _determineContactGroupId()


    /**
     * Ermittelt die User-ID zur übergebenen E-Mail-Adresse
     *
     * @param       null|string $email
     *
     * @return      int
     *
     * @version     1.1.0 / 2025-01-30
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _determineUserIdByEmail(null|string $email): int|null
    {
        if ($email === null) {
            return null;
        }
        $email = StringHelper::normalizeEmail($email);

        if (!isset($this->_users[$email])) {
            return null;
        }
        return $this->_users[$email]['id'];

    } // _determineUserIdByEmail()


    /**
     * Führt Aktionen nach dem Import aus
     *
     * @param       AfterImport $Event
     *
     * @version     1.2.0 / 2024-09-24
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function afterImport(AfterImport $Event): void
    {
        parent::afterImport($Event);

    } // afterImport()


    /**
     * Führt Aktionen vor dem Import aus
     *
     * @param       BeforeImport $Event
     *
     * @version     1.4.0 / 2025-01-30
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function beforeImport(BeforeImport $Event): void
    {
        parent::beforeImport($Event);

        $this->_ContactGroupService = new VisitContactGroupService();
        /*
        **  Liste der ContactGroups ermitteln und einen Array erzeugen, bei dem die Label als Schlüssel
        **  verwendet werden. */
        $ContactGroupCollection = VisitContactGroup::query()->get();

        $this->_contactGroups = [];

        foreach ($ContactGroupCollection as $ContactGroup) {
            $contactGroup = $ContactGroup->labels->de;

            $this->_contactGroups[$contactGroup] = $ContactGroup->toArray();
        }
        $RoleCollection = Role::query()->get();

        foreach ($RoleCollection as $Role) {
            $identifier = $Role->identifier;

            $this->_roles[$identifier] = $Role->toArray();
        }
        $this->_RoleUserService = new RoleUserService();
        $this->_UserService = new UserService();
        /*
        **  Liste der ContactGroups ermitteln und einen Array erzeugen, bei dem die Label als Schlüssel
        **  verwendet werden. */
        $UserCollection = User::query()->get();

        $this->_users = [];

        foreach ($UserCollection as $User) {
            $email = StringHelper::normalizeEmail($User->email);

            $this->_users[$email] = $User->toArray();
        }
        /*
        **  Vor dem Import werden alle Mitarbeiter deaktiviert. Die in der Importdatei enthaltenen
        **  Mitarbeiter, zu denen bereits Datensätze in der Datenbank existierten, werden dann beim
        **  Import wieder reaktiviert. */
        VisitEmployee::query()->update(['active' => 0]);

    } // beforeImport()


    /**
     * Diese Methode wird für jede fehlerhafte Zeile aufgerufen
     *
     * @param       Failure ...$Failures
     *
     * @requires    Maatwebsite\Excel\Concerns\SkipsOnFailure
     * @requires    Maatwebsite\Excel\Concerns\WithValidation
     *              Damit Felder mit der rules()-Methode validiert werden und im Fehlerfall die Methode aufgerufen wird.
     *              Alternativ kann auch noch der Trait Maatwebsite\Excel\Concerns\SkipsFailures eingebunden werden,
     *              der die Methode failures() zur Verfügung stellt, die eine Collection mit allen Fehlern liefert.
     *
     * @version     1.2.0 / 2024-09-24
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function onFailure(Failure ...$Failures): void
    {
        parent::onFailure(...$Failures);
        /*
        **  Ersten Fehler aus der Liste ermitteln.
        **  Die Zeilennummer und die Daten der Zeile werden von diesem Fehler verwendet. Sie werden nur einmalig
        **  benötigt. */
        $CurrentFailure = current($Failures);
        /*
        **  Werte der Zeile ermitteln und anschließend versuchen einen passenden Datensatz aus dem Bestand zu
        **  ermitteln. Wenn ein passender Eintrag gefunden wurde, dann werden die Fehler beim Datensatz gespeichert. */
        $values = $CurrentFailure->values();
        /*
        **  Query zusammenstellen, mit der ein zur aktuellen Zeile passender Datensatz aus der Datenbank ermittelt
        **  werden soll.
        **  Wenn die Abfrage ein Ergebnis liefert, dann wird der Mitarbeiter-Datensatz reaktiviert und die aufgetretenen
        **  Fehler werden beim Datensatz gespeichert. */
        $Query = VisitEmployee::query();

        foreach ($this->_uniqueBy() as $field) {
            $Query->where($field, '=', $values[$field]);
        }
        $Employee = $Query->first();

        if ($Employee !== null) {
            $Employee->active = 1;
            $Employee->import_failure_count = count($Failures);
            $Employee->import_failures = $Failures;

            $Employee->save();
        }
    } // onFailure()


    /**
     * Verarbeitet eine einzelne Zeile aus dem Import
     *
     * @param       Row $Row
     *              Die aktuelle Zeile als Row-Objekt.
     *
     * @return      void
     *
     * @version     1.9.0 / 2025-04-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function onRow(Row $Row): void
    {
        $row = $Row->toArray();

        $uniqueData = array_intersect_key($row, $this->_uniqueBy());
        /*
        **  Model aus der Datenbank lesen oder neues Objekt erzeugen. Dann das Model füllen und speichern. */
        $Employee = VisitEmployee::firstOrNew($uniqueData);

        $mobilePhoneNumber = StringHelper::explodePhoneNumber($row['mobile_phone_number']);
        $phoneNumber = StringHelper::explodePhoneNumber($row['phone_number']);

        $data = [
            'active'                           => 1,
            'contact'                          => $row['contact'],
            'contact_group_id'                 => $row['contact_group_id'],
            'department'                       => $row['department'],
            'email'                            => $row['email'],
            'emergency_contact'                => $row['emergency_contact'],
            'first_name'                       => $row['first_name'],
            'function'                         => $row['function'],
            'global_contact'                   => $row['global_contact'],
            'import_failure_count'             => 0,
            'import_failures'                  => [],
            'last_name'                        => $row['last_name'],
            'license_plate'                    => $row['license_plate'],
            'mobile_phone_number'              => $mobilePhoneNumber['phone-number'],
            'mobile_phone_number_country_code' => $mobilePhoneNumber['country-code'],
            'phone_number'                     => $phoneNumber['phone-number'],
            'phone_number_country_code'        => $phoneNumber['country-code'],
        ];
        if (empty($data['email'])) {
            $data['active'] = 0;
            $data['import_failure_count']++;
            $data['import_failures'][] = trans('visitor-management::employee-import.error.email-missing');
        }
        if ($data['contact'] === true || $data['emergency_contact'] === true || $data['global_contact'] === true) {
            if (empty($data['phone_number']) && empty($data['mobile_phone_number'])) {
                $data['active'] = 0;
                $data['import_failure_count']++;
                $data['import_failures'][] = trans('visitor-management::employee-import.error.phone-number-missing');
            }
        }
        if (empty($data['import_failures'])) {
            $data['import_failures'] = null;
        }
        try {
            if ($Employee->id === null) {
                $data['personnel_number'] = $row['personnel_number'];
            }
            $Employee->fill($data);
            /*
            **  Falls es bereits einen händisch erzeugten User mit der aktuellen Mail-Adresse gibt,
            **  dann wird dieser mit dem Mitarbeiter verknüpft. */
            if ($Employee->user_id === null && $Employee->email !== null) {
                $userId = $this->_determineUserIdByEmail($Employee->email);

                if ($userId !== null) {
                    $Employee->user_id = $userId;
                }
            }
            $Employee->save();

            $this->_incrementRowCountSaved();
            /*
            **  Das Flag "Create User Account" wird nur für die Erstanlage verwendet. Dass bestehende
            **  User-Accounts über den Import aktualisiert werden, ist aktuell nicht vorgesehen.  */
            if ($row['create_user_account'] === true && $Employee->user_id === null) {
                if (empty($row['email']) || empty($row['create_user_roles'])) {
                    /*
                    **  Neue Fehler können nicht direkt der Model-Eigenschaft zugewiesen werden. Deshalb muss
                    **  der Wert erst einer Variablen zugewiesen werden. Ansonsten kommt es zu folgendem Fehler:
                    **
                    **      Indirect modification of overloaded property has no effect
                    */
                    $failures = $Employee->import_failures;
                    $failureCount = $Employee->import_failure_count;

                    if (empty($row['email'])) {
                        $failures[] = trans('visitor-management::employee-import.user-error.email-missing');
                        $failureCount++;
                    }
                    if (empty($row['create_user_roles'])) {
                        $failures[] = trans('visitor-management::employee-import.user-error.roles-missing');
                        $failureCount++;
                    }
                    $Employee->import_failures = $failures;
                    $Employee->import_failure_count = $failureCount;

                    $Employee->save();

                } else {
                    /*
                    **  Neuen User-Account anlegen. */
                    $User = $this->_UserService->create([
                        'email' => $Employee->email,
                        'name' => $Employee->getFullName(),
                    ]);
                    $Employee->user_id = $User->id;

                    $Employee->save();
                    /*
                    **  Definierte Rollen verknüpfen. */
                    foreach ($row['create_user_roles'] as $roleId) {
                        $this->_RoleUserService->create([
                            'role_id' => $roleId,
                            'user_id' => $User->id,
                        ]);
                    }
                }
            }
        } catch (\Exception $E) {
            $this->_errors[] = [
                'code' => $E->getCode(),
                'message' => $E->getMessage(),
            ];
        }
    } // onRow()


    /**
     * Bereitet Werte einer Zeile vor der Validierung auf
     *
     * Als Schlüsselnamen müssen die aufbereiteten Spaltennamen (First Name > first_name) aus der Importdatei
     * verwendet werden.
     *
     * @param       array $data
     *              Die aktuelle Zeile als Array.
     *
     * @param       int $index
     *              Die Zeilennummer.
     *
     * @return      array
     *
     * @requires    Maatwebsite\Excel\Concerns\WithValidation
     *              Diese Methode wird automatisch in der Methode Maatwebsite\Excel\Row::toArray() aufgerufen. Das führt
     *              leider dazu, dass der Code jedes Mal ausgeführt wird, wenn toArray() verwendet wird. Da dabei aber
     *              immer das unveränderte Row-Objekt verwendet wird, kommt es glücklicherweise nicht dazu, dass Daten
     *              doppelt modifiziert werden.
     *
     * @version     1.7.0 / 2025-02-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function prepareForValidation($data, $index): array
    {
        $data = parent::prepareForValidation($data, $index);

        $data['import_failures'] = [];
        $data['import_failure_count'] = 0;

        $data['contact'] = empty($data['contact']) === true ? 0 : ($data['contact'] === 'yes' ? 1 : 0);
        $data['emergency_contact'] = empty($data['emergency_contact']) === true ? 0 : ($data['emergency_contact'] === 'yes' ? 1 : 0);
        $data['global_contact'] = empty($data['global_contact']) === true ? 0 : ($data['global_contact'] === 'yes' ? 1 : 0);
        $data['create_user_account'] = empty($data['create_user_account']) === true ? false : ($data['create_user_account'] === 'yes' ? true : false);
        /*
        **  Wenn der Mitarbeiter als globale Kontaktperson deklariert ist, dann wird "contact" auf FALSE gesetzt. Das
        **  erleichtert das Handling, weil nicht darauf geachtet werden muss, dass er nicht doppelt in der Auswahlliste
        **  erscheint.
        **  Die ContactGroup wird trotzdem gesetzt, sofern im Import eine vorhanden ist. */
        if ($data['global_contact'] === true) {
            $data['contact'] = false;
        }
        $data['contact_group_id'] = $this->_determineContactGroupID($data['contact_group']);
        $data['email'] = StringHelper::normalizeEmail($data['email']);
        $data['mobile_phone_number'] = StringHelper::normalizePhoneNumber($data['mobile_phone_number']);
        $data['phone_number'] = StringHelper::normalizePhoneNumber($data['phone_number']);

        if ($data['create_user_account'] === true && !empty($data['create_user_roles'])) {
            $roles = [];

            foreach (explode(',', str_replace(' ', '', trim($data['create_user_roles']))) as $role) {
                $identifier = ucfirst(strtolower(trim($role)));

                if (isset($this->_roles[$identifier])) {
                    $roles[] = $this->_roles[$identifier]['id'];
                }
            }
            if (empty($roles)) {
                $data['create_user_roles'] = null;
            } else {
                $data['create_user_roles'] = $roles;
            }
        }
        return $data;

    } // prepareForValidation()


    /**
     * Liefert ValidationRules
     *
     * In dieser Methode müssen die aufbereiteten Spaltennamen (First Name > first_name) aus der Importdatei
     * als Schlüssel verwendet werden.
     *
     * @return      array
     *
     * @requires    Maatwebsite\Excel\Concerns\WithValidation
     *              Damit die Methode rules() automatisch verwendet wird.
     *
     * @requires    Maatwebsite\Excel\Concerns\WithHeadingRow
     *              Um die Überschriften der Spalten als Schlüssel und auch in den Rules (z.B. "maximum" => "gte:*.minimum")
     *              verwenden zu können.
     *
     * @version     1.5.0 / 2025-01-07
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function rules(): array
    {
        return [
            'personnel_number'    => ['required', 'numeric'],
            'contact'             => ['required', 'boolean'],
            'contact_group_id'    => ['nullable', 'exists:contact_groups,id'],
            'department'          => ['nullable', 'max:50'],
            'email'               => ['nullable', 'max:255', 'email'],
            'emergency_contact'   => ['required', 'boolean'],
            'first_name'          => ['required', 'max:50'],
            'function'            => ['nullable', 'max:50'],
            'global_contact'      => ['required', 'boolean'],
            'last_name'           => ['required', 'max:50'],
            'license_plate'       => ['nullable', 'max:255'],
            'mobile_phone_number' => [new RequiredIfEmpty('phone_number', $this->_currentRowData)],
            'phone_number'        => [new RequiredIfEmpty('mobile_phone_number', $this->_currentRowData)],
        ];
    } // rules()


    /**
     * Liefert den Namen/Schlüssel der Spalte, die als identifizierendes Merkmal verwendet werden soll
     * *
     * @return      array|string
     *
     * @version     1.1.0 / 2024-09-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function uniqueBy(): array|string
    {
        /*
        **  Bei allen DBMS (außer SQL Server) muss das Feld zum identifizierenden Merkmal in der
        **  Datenbank noch zusätzlich mit einem Unique-Key versehen werden. */
        return 'personnel_number';

    } // uniqueBy()


    /**
     * Definiert die Spalten die bei einem Update aktualisiert werden sollen
     *
     * In dieser Methode müssen die originalen Feldnamen aus der Datenbank als Schlüssel verwendet werden.
     *
     * WICHTIG:
     * Die identifizierende Spalte (in diesem Fall "personnel_number") sollte hier nicht enthalten sein.
     *
     * @return      array
     *
     * @version     1.4.0 / 2025-01-07
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function upsertColumns(): array
    {
        return [
            'active',
            'contact',
            'contact_group_id',
            'department',
            'email',
            'emergency_contact',
            'first_name',
            'function',
            'global_contact',
            'import_failures',
            'import_failure_count',
            'last_name',
            'license_plate',
            'mobile_phone_number',
            'phone_number',
        ];
    } // upsertColumns()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class EmployeesImport extends BaseImport implements ... {}
