<?php
/**
 * Import Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-modules/visitor-management
 * @subpackage  Imports
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanModules\VisitorManagement\Imports;


use BplanBase\Globals\Helpers\StringHelper;
use BplanModules\VisitorManagement\Enums\ImportType;
use BplanModules\VisitorManagement\Models\VisitLanguage;
use BplanModules\VisitorManagement\Models\VisitVisitor;
use BplanModules\VisitorManagement\Services\VisitorService;
use Maatwebsite\Excel\Concerns\Importable;
use Maatwebsite\Excel\Concerns\OnEachRow;
use Maatwebsite\Excel\Concerns\WithUpsertColumns;
use Maatwebsite\Excel\Concerns\WithUpserts;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Events\AfterImport;
use Maatwebsite\Excel\Events\BeforeImport;
use Maatwebsite\Excel\Row;
use Maatwebsite\Excel\Validators\Failure;


/**
 * Import Class
 *
 * @version     1.3.0 / 2025-02-05
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class VisitorsImport extends BaseImport implements
    OnEachRow,
    WithUpsertColumns,
    WithUpserts,
    WithValidation
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    use Importable;


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string $_defaultLocale
     */
    protected $_defaultLocale;


    /**
     * @var     ImportType $_ImportType
     */
    protected $_ImportType = ImportType::Visitor;


    /**
     * @var     array $_languages
     */
    protected $_languages;


    /**
     * @var     array $_visitors
     */
    protected $_visitors;


    /**
     * @var     VisitorService $_VisitorService
     */
    protected $_VisitorService;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string $_subPath
     */
    protected static $_subPath = 'import/visitors';


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _initLanguages(): self
    {
        $Languages = Language::where('active', '=', 1)
            ->get();

        foreach ($Languages as $Language) {
            $code = $Language->code;

            $this->_languages[$code] = $Language;
        }
        return $this;

    } // _initLanguages()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _initVisitors(): self
    {
        $Visitors = Visitor::get();

        foreach ($Visitors as $Visitor) {
            if ($Visitor->email === null) {
                /*
                **  Eine eindeutige Zuordnung ist nur dann möglich, wenn sowohl Vorname und Nachname als auch
                **  E-Mail-Adresse vorhanden sind. Ohne E-Mail wird der Besucher nicht in die Vergleichsdaten
                **  aufgenommen. */
                continue;
            }
            $key = mb_strtolower($Visitor->first_name).'.'.mb_strtolower($Visitor->last_name).'.'.mb_strtolower($Visitor->email);

            $this->_visitors[$key] = $Visitor;
        }
        return $this;

    } // _initVisitors()


    /**
     * Führt Aktionen nach dem Import aus
     *
     * @param       AfterImport $Event
     *
     * @version     1.0.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function afterImport(AfterImport $Event): void
    {
        parent::afterImport($Event);

    } // afterImport()


    /**
     * Führt Aktionen vor dem Import aus
     *
     * @param       BeforeImport $Event
     *
     * @version     1.0.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function beforeImport(BeforeImport $Event): void
    {
        parent::beforeImport($Event);

        $this->_VisitorService = new VisitorService();

        $this->_defaultLocale = config('app.fallback_locale');

        $this->_initLanguages();
        $this->_initVisitors();

    } // beforeImport()


    /**
     * Diese Methode wird für jede fehlerhafte Zeile aufgerufen
     *
     * @param       Failure ...$Failures
     *
     * @requires    Maatwebsite\Excel\Concerns\SkipsOnFailure
     * @requires    Maatwebsite\Excel\Concerns\WithValidation
     *              Damit Felder mit der rules()-Methode validiert werden und im Fehlerfall die Methode aufgerufen wird.
     *              Alternativ kann auch noch der Trait Maatwebsite\Excel\Concerns\SkipsFailures eingebunden werden,
     *              der die Methode failures() zur Verfügung stellt, die eine Collection mit allen Fehlern liefert.
     *
     * @version     1.0.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function onFailure(Failure ...$Failures): void
    {
        parent::onFailure(...$Failures);
        /*
        **  Ersten Fehler aus der Liste ermitteln.
        **  Die Zeilennummer und die Daten der Zeile werden von diesem Fehler verwendet. Sie werden nur einmalig
        **  benötigt. */
        $CurrentFailure = current($Failures);
        /*
        **  Werte der Zeile ermitteln und anschließend versuchen einen passenden Datensatz aus dem Bestand zu
        **  ermitteln. Wenn ein passender Eintrag gefunden wurde, dann werden die Fehler beim Datensatz gespeichert. */
        $values = $CurrentFailure->values();
        /*
        **  Query zusammenstellen, mit der ein zur aktuellen Zeile passender Datensatz aus der Datenbank ermittelt
        **  werden soll.
        **  Wenn die Abfrage ein Ergebnis liefert, dann wird der Mitarbeiter-Datensatz reaktiviert und die aufgetretenen
        **  Fehler werden beim Datensatz gespeichert. */
        $Query = Visitor::query();

        foreach ($this->_uniqueBy() as $field) {
            $Query->where($field, '=', $values[$field]);
        }
        $Visitor = $Query->first();

        if ($Visitor !== null) {
            $Visitor->active = 1;
            $Visitor->import_failure_count = count($Failures);
            $Visitor->import_failures = $Failures;

            $Visitor->save();
        }
    } // onFailure()


    /**
     * Verarbeitet eine einzelne Zeile aus dem Import
     *
     * @param       Row $Row
     *              Die aktuelle Zeile als Row-Objekt.
     *
     * @return      void
     *
     * @version     1.2.0 / 2025-02-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function onRow(Row $Row): void
    {
        $row = $Row->toArray();

        $uniqueData = array_intersect_key($row, $this->_uniqueBy());
        /*
        **  Model aus der Datenbank lesen oder neues Objekt erzeugen. Dann das Model füllen und speichern. */
        $Visitor = Visitor::firstOrNew($uniqueData);

        $mobilePhoneNumber = StringHelper::explodePhoneNumber($row['mobile_phone_number']);

        $data = [
            'active'                           => 1,
            'company'                          => $row['company'],
            'email'                            => $row['email'],
            'first_name'                       => $row['first_name'],
            'import_failure_count'             => 0,
            'import_failures'                  => [],
            'last_name'                        => $row['last_name'],
            'mobile_phone_number'              => $mobilePhoneNumber['phone-number'],
            'mobile_phone_number_country_code' => $mobilePhoneNumber['country-code'],
            'preferred_language'               => $row['preferred_language'],
        ];
        if (empty($data['import_failures'])) {
            $data['import_failures'] = null;
        }
        try {
            $Visitor->fill($data);

            $Visitor->save();

            $this->_incrementRowCountSaved();

        } catch (\Exception $E) {
            $this->_errors[] = [
                'code' => $E->getCode(),
                'index' => $Row->getIndex(),
                'message' => $E->getMessage(),
            ];
        }
    } // onRow()


    /**
     * Bereitet Werte einer Zeile vor der Validierung auf
     *
     * Als Schlüsselnamen müssen die aufbereiteten Spaltennamen (First Name > first_name) aus der Importdatei
     * verwendet werden.
     *
     * @param       array $data
     *              Die aktuelle Zeile als Array.
     *
     * @param       int $index
     *              Die Zeilennummer.
     *
     * @return      array
     *
     * @requires    Maatwebsite\Excel\Concerns\WithValidation
     *              Diese Methode wird automatisch in der Methode Maatwebsite\Excel\Row::toArray() aufgerufen. Das führt
     *              leider dazu, dass der Code jedes Mal ausgeführt wird, wenn toArray() verwendet wird. Da dabei aber
     *              immer das unveränderte Row-Objekt verwendet wird, kommt es glücklicherweise nicht dazu, dass Daten
     *              doppelt modifiziert werden.
     *
     * @version     1.2.0 / 2025-02-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function prepareForValidation($data, $index): array
    {
        $data = parent::prepareForValidation($data, $index);

        $data['import_failures'] = [];
        $data['import_failure_count'] = 0;

        $data['email'] = StringHelper::normalizeEmail($data['email']);
        $data['mobile_phone_number'] = StringHelper::normalizePhoneNumber($data['mobile_phone_number']);

        if (!empty($data['preferred_language'])) {
            $data['preferred_language'] = strtolower($data['preferred_language']);

            if (!isset($this->_languages[$data['preferred_language']])) {
                $data['preferred_language'] = $this->_defaultLocale;
            }
        } else {
            $data['preferred_language'] = $this->_defaultLocale;
        }
        return $data;

    } // prepareForValidation()


    /**
     * Liefert ValidationRules
     *
     * In dieser Methode müssen die aufbereiteten Spaltennamen (First Name > first_name) aus der Importdatei
     * als Schlüssel verwendet werden.
     *
     * @return      array
     *
     * @requires    Maatwebsite\Excel\Concerns\WithValidation
     *              Damit die Methode rules() automatisch verwendet wird.
     *
     * @requires    Maatwebsite\Excel\Concerns\WithHeadingRow
     *              Um die Überschriften der Spalten als Schlüssel und auch in den Rules (z.B. "maximum" => "gte:*.minimum")
     *              verwenden zu können.
     *
     * @version     1.0.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function rules(): array
    {
        return [
            'company'             => ['nullable', 'max:100'],
            'email'               => ['nullable', 'max:255', 'email'],
            'first_name'          => ['required', 'max:50'],
            'last_name'           => ['required', 'max:50'],
            'mobile_phone_number' => ['nullable', 'max:255'],
            'preferred_language'  => ['nullable', 'max:2'],
        ];
    } // rules()


    /**
     * Liefert den Namen/Schlüssel der Spalte, die als identifizierendes Merkmal verwendet werden soll
     * *
     * @return      array|string
     *
     * @version     1.0.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function uniqueBy(): array|string
    {
        return [
            'email',
            'first_name',
            'last_name',
        ];

    } // uniqueBy()


    /**
     * Definiert die Spalten die bei einem Update aktualisiert werden sollen
     *
     * In dieser Methode müssen die originalen Feldnamen aus der Datenbank als Werte verwendet werden.
     *
     * @return      array
     *
     * @version     1.0.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function upsertColumns(): array
    {
        return [
            'active',
            'company',
            'email',
            'first_name',
            'import_failures',
            'import_failure_count',
            'last_name',
            'mobile_phone_number',
            'preferred_language',
        ];
    } // upsertColumns()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class VisitorsImport extends BaseImport implements ... {}
