<?php
/**
 * Notification Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-modules/visitor-management
 * @subpackage  Notifications
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanModules\VisitorManagement\Notifications;


use BplanComponents\ICS\ICS\Event;
use BplanComponents\ICS\ICS\ICSFile;
use BplanModules\VisitorManagement\Enums\VisitTypeIdentifier;
use BplanModules\VisitorManagement\Models\VisitAppointment;
use BplanModules\VisitorManagement\Models\VisitType;
use BplanModules\VisitorManagement\Repositories\VisitTypeRepository;
use chillerlan\QRCode\QRCode;
use chillerlan\QRCode\QROptions;
use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\HtmlString;


/**
 * Notification Class
 *
 * Benachrichtigung an einen Besucher (Hauptbesucher) über die Löschung eines Termins (per E-Mail).
 *
 * @version     1.2.0 / 2025-03-18
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class VisitorAppointmentCanceled extends Notification
{

/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */

    use Queueable;


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     VisitAppointment $_Appointment
     */
    private $_Appointment;


    /**
     * @var     array $_attachmentFileTypes
     */
    private $_attachmentFileTypes = [
        // AttachmentFileType::HouseRules,
        AttachmentFileType::RouteDescription,
    ];


    /**
     * @var     VisitAppointment $_BulkAppointment
     */
    private $_BulkAppointment;


    /**
     * @var     string $_company
     */
    private $_company;


    /**
     * @var     User $_CreateUser
     */
    private $_CreateUser;


    /**
     * @var     string $_defaultLanguage
     */
    private $_defaultLanguage;


    /**
     * @var     ModelEvent $_Event
     */
    private $_Event;


    /**
     * @var     array $_languages
     */
    private $_languages;


    /**
     * @var     VisitEmployee $_MainEmployee
     */
    private $_MainEmployee;


    /**
     * @var     User|VisitEmplyoee $_Organizer
     */
    private $_Organizer;


    /**
     * @var     VisitVisitor $_Visitor
     */
    private $_Visitor;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Create a new notification instance.
     *
     * @param       VisitAppointment $Appointment
     *
     * @param       ModelEvent $Event
     *
     * @version     1.0.0 / 2025-01-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(VisitAppointment $Appointment, ModelEvent $Event)
    {
        $this->_Appointment = $Appointment;
        $this->_Event = $Event;

        if ($this->_Appointment->bulk_appointment_id !== null) {
            $this->_BulkAppointment = VisitAppointmentRepository::getById($this->_Appointment->bulk_appointment_id);
        }
        $this->_CreateUser = UserRepository::getById($this->_Appointment->create_user_id);
        $this->_Visitor = VisitVisitorRepository::getById($this->_Appointment->main_visitor_id);

        $this->_Organizer = $this->_CreateUser;

        if ($this->_Appointment->main_employee_id !== null) {
            $this->_MainEmployee = VisitEmployeeRepository::getById($this->_Appointment->main_employee_id);
        }
        if (!empty($this->_Appointment->organizer_employee_id)) {
            $this->_Organizer = VisitEmployeeRepository::getById($this->_Appointment->organizer_employee_id);
        }
        $this->_company = config('project.company');

        list($this->_defaultLanguage) = config('app.fallback_locale', 'en');
        /*
        **  Sprachen zusammenstellen.
        **  Unabhängig von den tatsächlich verfügbaren Sprachen, wird hier lediglich die bevorzugte Sprache des Besuchers
        **  und die Standardsprache des Systems abgearbeitet. Der zusammengestellte Array wird noch bereinigt, für den
        **  Fall dass beide Sprachen identisch sind. */
        $this->_languages = array_unique([
            $this->_Visitor->preferred_language,
            $this->_defaultLanguage,
        ]);
    } // __construct()


    /**
     *
     * @param       User|VisitEmployee|VisitVisitor $Notifiable
     *
     * @param       null|string & $icsContents
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2025-01-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _buildIcsFile(object $Notifiable, null|string & $icsContents = null): string
    {
        $lang = $this->_Visitor->preferred_language;
        /*
        **  Event für ics-Datei zusammenstellen.
        **
        **  Da die verwendete ICS-Klasse keine Möglichkeit bietet eine Timezone zu definieren, muss die
        **  Timezone für Beginn und Ende nach "UTC" geändert werden. */
        $eventDetails = [
            'description' => $this->_Appointment->subject,
            'dtstart' => $this->_Appointment->valid_from->setTimezone('UTC')->format('Y-m-d H:i'),
            'dtend' => $this->_Appointment->valid_until->setTimezone('UTC')->format('Y-m-d H:i'),
            'location' => $this->_company['full_name'].', '.$this->_company['address'],
            'sequence' => 0,
            'summary' => $this->_Appointment->subject,
            'uid' => $this->_Appointment->uuid,
        ];
        $Event = new Event($eventDetails);
        /*
        **  Organizer ist entweder der Organizer des Termins oder der erstellende User und der erste ist Attendee
        **  immer der verantwortliche Mitarbeiter, sofern dem Termin einer zugeordnet wurde. Anschließend folgt
        **  der Hauptbesucher des Termins. */
        $Event->setOrganizer($this->_Organizer->getFullName(), $this->_Organizer->email);

        if ($this->_MainEmployee !== null) {
            /*
            **  Wenn ein Mitarbeiter verknüpft ist, dann wird dieser als erster zusätzlicher Attendee hinzugefügt. */
            $Event->addAttendee($this->_MainEmployee->getFullName(), $this->_MainEmployee->email, accepted: true);
        }
        $Event->addAttendee($this->_Visitor->getFullName(), $this->_Visitor->email ?? trans('notifications.no-mail'), accepted: true);
        /*
        **  Dieser Fall tritt ein wenn es sich bei dem Appointment um einen Einzelbesuch zu einem "BulkAppointment" handelt.
        **  In diesem Fall sind Basisdaten zu Besuchern im visitors-Feld des BulkAppointments gespeichert.
        **  Alternativ könnten die zugehörigen Besucher auch aus der Datenbank gelesen werden, indem die Hauptbesucher
        **  aller Einzeltermine ermittelt werden. Für den konkreten Fall sind die beim Appointment gespeicherten Daten
        **  aber vollkommen ausreichend. */
        if ($this->_BulkAppointment !== null) {
            foreach ($this->_BulkAppointment->visitors as $Visitor) {
                /*
                **  Der Hauptbesucher, dessen Daten ebenfalls in "visitors" gespeichert sind, wird übersprungen. */
                if ((int) $Visitor->id === $this->_Visitor->id) {
                    continue;
                }
                $Event->addAttendee($Visitor->first_name.' '.$Visitor->last_name, $Visitor->email ?? trans('notifications.no-mail'));
            }
        }
        /*
        **  ics-Datei erstellen und speichern. */
        $IcsFile = new ICSFile();

        $IcsFile->addEvent($Event);
        $IcsFile->setLanguage($lang);
        $IcsFile->setMethodCancel();

        $storagePath = storage_path('app/tmp');

        if (!file_exists($storagePath)) {
            mkdir($storagePath, 0777, true);
        }
        $icsFile = $storagePath.'/'.$this->_Appointment->uuid.'.ics';
        $icsContents = $IcsFile->toString();

        file_put_contents($icsFile, $icsContents);

        return $icsFile;

    } // _buildIcsFile()


    /**
     * Get the array representation of the notification.
     *
     * @param       User|VisitEmployee|VisitVisitor $Notifiable
     *
     * @return      array<string, mixed>
     *
     * @version     1.0.0 / 2025-01-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function toArray(object $Notifiable): array
    {
        return [
            //
        ];
    } // toArray()


    /**
     * Get the mail representation of the notification.
     *
     * @param       User|VisitEmployee|VisitVisitor $Notifiable
     *
     * @throws      Exception
     *
     * @return      MailMessage
     *
     * @version     1.2.0 / 2025-03-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function toMail(object $Notifiable): MailMessage
    {
        $lang = $this->_Visitor->preferred_language;
        /*
        **  VisitType zum Appointment ermitteln. */
        $VisitType = VisitTypeRepository::getByVisitReason($this->_Appointment->visit_reason_id);

        $hostPhone = null;

        if ($this->_MainEmployee !== null) {
            $this->_MainEmployee = VisitEmployeeRepository::getById($this->_Appointment->main_employee_id);

            $host = $this->_MainEmployee->getFullName();
            $hostPhone = $this->_MainEmployee->getPhoneNumber() ?? $this->_MainEmployee->getPhoneNumber(getMobile: true);

        } else {
            $host = $this->_CreateUser->getFullName();
        }
        $lc_dateTimeFormat = trans('globals::global.datetime.format-+', [], $lang);

        $lc_details = trans('notifications.appointment-saved.mail.details', [
            'host' => $host,
            'subject' => $this->_Appointment->subject,
            'time-range' => $this->_Appointment->valid_from->format($lc_dateTimeFormat).' - '.$this->_Appointment->valid_until->format($lc_dateTimeFormat),
        ], $lang);

        $lc_intro = trans('notifications.appointment-canceled.mail.intro', [], $lang);

        $MailMessage = (new MailMessage)
            ->subject(trans('notifications.appointment-canceled.mail.subject', ['date' => $this->_Appointment->valid_from->format($lc_dateTimeFormat)], $lang))
            ->greeting(trans('notifications.greeting.user', ['username' => $this->_Visitor->getFullName()], $lang))
            ->line($lc_intro)
            ->line(new HtmlString($lc_details));

        if($this->_Appointment->visitType->visit_group === VisitTypeGroupIdentifier::Logistics->value) {
            $MailMessage->line(trans('notifications.appointment-saved.mail.appointment-order-number', ['order-number' => $this->_Appointment->order_number], $lang));
        }
        if ($hostPhone !== null) {
            $MailMessage->line(trans('notifications.appointment-saved.mail.call-us', ['phone' => $hostPhone], $lang));
        }
        $MailMessage->salutation(trans('notifications.appointment-saved.mail.salutation.employee', ['host' => $host], $lang));

        $icsFile = $this->_buildIcsFile($Notifiable);

        $MailMessage->attach($icsFile, [
            'as' => 'invite.ics',
            'mime' => 'application/ics',
        ]);

        Log::notice('Sending visitor delete-notification to "'.$Notifiable->getFullName().'" ('.$Notifiable->email.', type '.basename(get_class($Notifiable)).') in '.__METHOD__.'().');

        return $MailMessage;

    } // toMail()


    /**
     * Get the notification's delivery channels.
     *
     * @param       User|VisitEmployee|VisitVisitor $Notifiable
     *
     * @return      array<int, string>
     *
     * @version     1.0.0 / 2025-01-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function via(object $Notifiable): array
    {
        return ['mail'];

    } // via()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class VisitorAppointmentCanceled extends Notification {}
