<?php
/**
 * Service Class
 *
 * Wegen individueller Anpassungen aus der automatischen Generierung genommen.
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @generated   2025-04-13 18:30:43
 * @package     bplan-modules/visitor-management
 * @subpackage  Services
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanModules\VisitorManagement\Services;


use App\Models\VisitAppointment as DerivedVisitAppointment;
use App\Repositories\UserRepository;
use BplanBase\Globals\Enums\ModelEvent;
use BplanModules\VisitorManagement\Actions\VisitAppointment\AnonymizeBulkVisitors;
use BplanModules\VisitorManagement\Actions\VisitAppointment\AnonymizeGroupVisitors;
use BplanModules\VisitorManagement\Actions\VisitAppointment\CheckOutOpen;
use BplanModules\VisitorManagement\Actions\VisitAppointment\Create;
use BplanModules\VisitorManagement\Actions\VisitAppointment\Delete;
use BplanModules\VisitorManagement\Actions\VisitAppointment\Update;
use BplanModules\VisitorManagement\Actions\VisitAppointment\FinishPast;
use BplanModules\VisitorManagement\Actions\VisitAppointment\UpdateProcessStatus;
use BplanModules\VisitorManagement\Enums\ProcessStatus;
use BplanModules\VisitorManagement\Enums\VisitReasonDefaultIdentifier;
use BplanModules\VisitorManagement\Enums\VisitTypeIdentifier;
use BplanModules\VisitorManagement\Models\VisitAppointment;
use BplanModules\VisitorManagement\Models\VisitAppointmentEmployee;
use BplanModules\VisitorManagement\Notifications\ApproachLoadingRamp;
use BplanModules\VisitorManagement\Notifications\EmployeeAccessAuthorizationCreated;
use BplanModules\VisitorManagement\Notifications\EmployeeAccessAuthorizationDeleted;
use BplanModules\VisitorManagement\Notifications\EmployeeAppointmentCanceled;
use BplanModules\VisitorManagement\Notifications\EmployeeAppointmentSaved;
use BplanModules\VisitorManagement\Notifications\EmployeeVisitorArrived;
use BplanModules\VisitorManagement\Notifications\VisitorAppointmentCanceled;
use BplanModules\VisitorManagement\Notifications\VisitorAppointmentSaved;
use BplanModules\VisitorManagement\Repositories\VisitAppointmentRepository;
use BplanModules\VisitorManagement\Repositories\VisitEmployeeRepository;
use BplanModules\VisitorManagement\Repositories\VisitReasonRepository;
use BplanModules\VisitorManagement\Repositories\VisitTypeRepository;
use BplanModules\VisitorManagement\Repositories\VisitVisitorRepository;
use BplanModules\VisitorManagement\Services\VisitAppointmentEmployeeService;
use Illuminate\Support\Collection;


/**
 * Service Class
 *
 * @version     1.1.0 / 2025-04-13
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class VisitAppointmentService
{

/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     array $_visitorProperties
     */
    private $_visitorProperties = [
        'id',
        'anonymized',
        'company',
        'email',
        'first_name',
        'last_name',
        'mobile_phone_number',
        'mobile_phone_number_country_code',
    ];


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Verarbeitet die übergebenen Visitor-IDs
     *
     * Zu den übergebenen IDs werden die Visitors aus der Datenbank gelesen. Mit den ermittelten Daten wird ein
     * Array erzeugt, der dann im visitors-Feld des Master-Datensatzes zum Massentermin gespeichert werden kann.
     *
     * @param       array $visitorIds<int, int>
     *              IDs von Visitors aus der Datenbank.
     *
     * @return 	    array
     *
     * @version     1.1.0 / 2025-03-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processBulkVisitors(array $visitorIds): array
    {
        $intersectKeys = array_combine($this->_visitorProperties, $this->_visitorProperties);
        $mergeKeys = array_fill_keys($this->_visitorProperties, null);

        if (!empty($visitorIds)) {
            if (empty($visitors)) {
                $visitors = [];
            }
            /*
            **  Datensätze zu den IDs aus der Datenbank lesen. */
            $VisitorCollection = VisitVisitorRepository::getCollectionByIds($visitorIds);
            /*
            **  Unerwünschte Felder entfernen und Datenaufbau vereinheitlichen. */
            foreach ($VisitorCollection as $Visitor) {
                $visitor = array_merge($mergeKeys, array_intersect_key($Visitor->toArray(), $intersectKeys));

                $visitor['id'] = (string) $visitor['id'];
                $visitor['anonymized'] = 0;

                $visitors[] = $visitor;
            }
        }
        return $visitors;

    } // _processBulkVisitors()


    /**
     * Vereinheitlicht und vereint Visitor-Datensätze
     *
     * @param       array $visitors<int, array>
     *              Ist bei der Neuanlage leer.
     * *            Erst bei der Nachbearbeitung in der Administration oder in der Personalisierung
     *              sind hier Datensätze enthalten (anonyme Besucher).
     *
     * @return 	    array
     *
     * @version     1.1.0 / 2025-03-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processGroupVisitors(array $visitors): array
    {
        $intersectKeys = array_combine($this->_visitorProperties, $this->_visitorProperties);
        $mergeKeys = array_fill_keys($this->_visitorProperties, null);

        if (!empty($visitors)) {
            /*
            **  Die Elemente zu den anonymen Visitors haben standardmäßig keine ID. Zur Unterscheidung
            **  von den angehängten Visitors aus der Datenbank, werden sie mit einer negativen ID
            **  versehen. */
            $fakeId = 0;
            /*
            **  Niedrigste ID aus den bereits vorhandenen Daten ermitteln. */
            $ids = array_column($visitors, 'id');

            if (!empty($ids)) {
                $min = min($ids);

                if ($min < 0) {
                    $fakeId = $min;
                }
            }
            foreach ($visitors as & $visitor) {
                $visitor = (array) $visitor;

                if (empty($visitor['id'])) {
                    /*
                    **  Elemente ohne ID erhalten die nächste verfügbare negative ID. */
                    $visitor['id'] = (string) --$fakeId;
                }
                $visitor['anonymized'] = 0;

                $visitor = array_merge($mergeKeys, array_intersect_key($visitor, $intersectKeys));
            }
        }
        return $visitors;

    } // _processGroupVisitors()


    /**
     * Sendet Notifications an Besucher, Mitarbeiter und User
     *
     * Diese Methode enthält die Logik, die entscheidet, welche in einen Termin involvierten Personen
     * benachrichtigt werden sollen.
     *
     * @param       VisitAppointment|DerivedVisitAppointment $Appointment
     *
     * @param       ModelEvent $Event
     *
     * @param       VisitAppointment|DerivedVisitAppointment|null $ForBulkAppointment
     *
     * @return 	    $this
     *
     * @version     2.1.0 / 2025-01-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _sendNotification(VisitAppointment|DerivedVisitAppointment $Appointment, ModelEvent $Event, VisitAppointment|null $ForBulkAppointment = null): self
    {
        /*
        **  VisitType zum Appointment ermitteln. */
        $VisitType = VisitTypeRepository::getByVisitReason($Appointment->visit_reason_id);

        if (!empty($Appointment->main_employee_id)) {
            /*
            **  Notification an den verantwortlichen Mitarbeiter senden. */
            $Employee = VisitEmployeeRepository::getById($Appointment->main_employee_id);

            if (!empty($Employee->email)) {
                if ($Event === ModelEvent::Deleted) {
                    if ($VisitType->identifier === VisitTypeIdentifier::AccessAuthorization->value) {
                        $Employee->notify(new EmployeeAccessAuthorizationDeleted($Appointment));

                    } elseif ($ForBulkAppointment !== null) {
                        $Employee->notify(new EmployeeAppointmentCanceled($ForBulkAppointment, ModelEvent::Updated));

                    } else {
                        if ($Appointment->bulk_appointment_id === null) {
                            $Employee->notify(new EmployeeAppointmentCanceled($Appointment, $Event));
                        }
                    }
                } else {
                    if ($VisitType->identifier === VisitTypeIdentifier::AccessAuthorization->value) {
                        $Employee->notify(new EmployeeAccessAuthorizationCreated($Appointment));

                    } elseif ($ForBulkAppointment !== null) {
                        /*
                        **  Bei einem nachträglichen Create zu einem BulkAppointment wird das zugehörige Objekt
                        **  übergeben. Die Benachrichtigung an den Mitarbeiter erfolgt zu diesem Appointment. */
                        $Employee->notify(new EmployeeAppointmentSaved($ForBulkAppointment, ModelEvent::Updated));

                    } else {
                        /*
                        **  Der Mitarbeiter erhält nur zu "normalen" Terminen und zum Master-Termin eines Bulk
                        **  eine Nachricht. Einzeltermine zu Bulk-Appointments werden vom Versand ausgeschlossen. */
                        if ($Appointment->bulk_appointment_id === null) {
                            $Employee->notify(new EmployeeAppointmentSaved($Appointment, $Event));
                        }
                    }
                }
            }
        }
        /*
        **  Wenn es sich beim aktuellen Termin um den Master-Termin eines Bulks handelt, dann werden dazu
        **  keine Mails an den Organizer, den Visitor oder den Create-User gesendet. */
        if ($VisitType->identifier === VisitTypeIdentifier::BulkAppointment->value) {
            return $this;
        }
        if ($Event === ModelEvent::Deleted) {
            if (!empty($Appointment->organizer_employee_id)) {
                $OrganizerEmployee = VisitEmployeeRepository::getById($Appointment->organizer_employee_id);

                $OrganizerEmployee->notify(new VisitorAppointmentCanceled($Appointment, $Event));

            } elseif (!empty($Appointment->main_visitor_id)) {
                $Visitor = VisitVisitorRepository::getById($Appointment->main_visitor_id);

                if (!empty($Visitor->email)) {
                    $Visitor->notify(new VisitorAppointmentCanceled($Appointment, $Event));

                } else {
                    $CreateUser = UserRepository::getById($Appointment->create_user_id);

                    if (strtolower($CreateUser->name) !== 'system') {
                        $CreateUser->notify(new VisitorAppointmentCanceled($Appointment, $Event));
                    }
                }
            }
        } else {
            if (!empty($Appointment->organizer_employee_id)) {
                /*
                **  Wenn ein Organizer ausgewählt ist, dann erhält dieser anstelle des Besuchers die
                **  Einladungs-Mail. */
                $OrganizerEmployee = VisitEmployeeRepository::getById($Appointment->organizer_employee_id);

                $OrganizerEmployee->notify(new VisitorAppointmentSaved($Appointment, $Event));

            } elseif (!empty($Appointment->main_visitor_id)) {
                /*
                **  Notification an den Hauptbesucher senden, sofern eine E-Mail-Adresse hinterlegt ist. */
                $Visitor = VisitVisitorRepository::getById($Appointment->main_visitor_id);

                if (!empty($Visitor->email)) {
                    $Visitor->notify(new VisitorAppointmentSaved($Appointment, $Event));

                } else {
                    /*
                    **  Wenn es keinen Organisator gibt und der aktuelle Besucher keine Mail-Adresse hat, dann
                    **  erhält der Ersteller des Termins die Einladung, außer es handelt sich um den System-User. */
                    $CreateUser = UserRepository::getById($Appointment->create_user_id);

                    if (strtolower($CreateUser->name) !== 'system') {
                        $CreateUser->notify(new VisitorAppointmentSaved($Appointment, $Event));
                    }
                }
            }
        }
        return $this;

    } // _sendNotification()


    /**
     * Anonymisiert die Bulk-Visitors mit den übergebenen IDs
     *
     * @param       array $visitorIds
     *
     * @return      int
     *
     * @version     1.0.0 / 2025-03-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function anonymizeBulkVisitors(array $visitorIds): int
    {
        if (empty($visitorIds)) {
            return 0;
        }
        $Action = new AnonymizeBulkVisitors();

        return $Action->execute($visitorIds);

    } // anonymizeBulkVisitors()


    /**
     * Anonymisiert die Visitors bei GroupVisits
     *
     * @param       array|Collection $appointments
     *              Ein Array mit Appoinment-IDs oder eine Collection von Appointments.
     *
     * @return      int
     *
     * @version     1.0.0 / 2025-03-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function anonymizeGroupVisitors(array|Collection $appointments): int
    {
        $Action = new AnonymizeGroupVisitors();

        return $Action->execute($appointments);

    } // anonymizeGroupVisitors()


    /**
     *
     * @param       int|string|VisitAppointment|DerivedVisitAppointment $appointment
     *
     * @param       string $loadingRampId
     *
     * @param       bool $updateProcessStatus
     *
     * @return 	    void
     *
     * @version     3.1.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function assignLoadingRamp(int|string|VisitAppointment $appointment, string $loadingRampId, bool $updateProcessStatus = true)
    {
        if ($appointment instanceof VisitAppointment) {
            $Appointment = $appointment;
        } else {
            $Appointment = VisitAppointmentRepository::getById($appointment);
        }
        $Visitor = VisitVisitorRepository::getById($Appointment->main_visitor_id);

        if ($updateProcessStatus === true) {
            $this->setProcessStatusCalledIn($Appointment, [
                'loading_ramp_id' => $loadingRampId,
            ]);
        } else {
            $this->update($Appointment, [
                'loading_ramp_id' => $loadingRampId,
            ]);
        }
        $Visitor->notify(new ApproachLoadingRamp($Appointment));

        return $Appointment;

    } // assignLoadingRamp()


    /**
     * Führt einen CheckOut bei allen offenen Terminen durch
     *
     * @return      int Liefert die Anzahl der betroffenen Datensätze.
     *
     * @version     1.0.0 / 2025-03-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function checkOutOpen(): int
    {
        $Action = new CheckOutOpen();

        return $Action->execute();

    } // checkOutOpen()


    /**
     *
     * @param       array $attributes
     *              Wenn der Schlüssel "visitors" in $attributes verwendet wird, dann wird dort ein assoziativer
     *              Array mit den Feldern "company", "email", "first_name", "last_name" und "mobile_phone_number"
     *              erwartet. Fehlende Schlüssel werden ergänzt und mit dem Wert NULL belegt.
     *              Der Schlüssel "visitors" wird ignoriert/überschrieben, wenn $visitorIDs nicht leer ist.
     *
     * @param       array $visitorIds
     *              Erwartet einen Array mit Visitor-IDs.
     *              Die Methoden ermittelt alle Visitors zu den übergebenen IDs und speichert deren Daten im
     *              Feld "visitors". Wenn dieser Parameter verwendet wird, dann wird ein eventuell in $attributes
     *              enthaltener Schlüssel "visitors" ignoriert.
     *
     * @param       VisitAppointment|DerivedVisitAppointment|null $ForBulkAppointment
     *
     * @return      VisitAppointment|DerivedVisitAppointment
     *
     * @version     3.1.0 / 2025-02-10
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function create(array $attributes, array $visitorIds = [], VisitAppointment|DerivedVisitAppointment|null $ForBulkAppointment = null): VisitAppointment|DerivedVisitAppointment
    {
        $VisitType = VisitTypeRepository::getByVisitReason($attributes['visit_reason_id']);

        if ($VisitType->identifier === VisitTypeIdentifier::BulkAppointment->value && !empty($visitorIds)) {
            $attributes['visitors'] = $this->_processBulkVisitors($visitorIds);

        } elseif ($VisitType->identifier === VisitTypeIdentifier::GroupVisit->value && !empty($attributes['visitors'])) {
            $attributes['visitors'] = $this->_processGroupVisitors($attributes['visitors']);
        }
        $Action = new Create();

        $Model = $Action->execute($attributes);

        $this->_sendNotification($Model, ModelEvent::Created, $ForBulkAppointment);

        return $Model;

    } // create()


    /**
     *
     * @param       array $attributes
     *
     * @param       array $visitorIds
     *              Erwartet einen Array mit Visitor-IDs.
     *              Die Methode ermittelt alle Visitors zu den übergebenen IDs und erzeugt neben dem Massentermin
     *              noch einen verknüpften Einzeltermin für jeden Visitor.
     *
     * @return      VisitAppointment|DerivedVisitAppointment Liefert das Model zum Massentermin.
     *
     * @version     1.4.0 / 2025-01-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function createBulk(array $attributes, array $visitorIds): VisitAppointment|DerivedVisitAppointment
    {
        $BulkAppointmentReason = VisitReasonRepository::getModelByIdentifier(VisitReasonDefaultIdentifier::BulkAppointment);
        $SingleVisitReason = VisitReasonRepository::getById($attributes['visit_reason_id']);

        $bulkAttributes = $attributes;

        $bulkAttributes['bulk_informations'] = [
            'visit_reason_id' => $attributes['visit_reason_id'],
            'visit_type_id' => $SingleVisitReason->visit_type_id,
            'visit_type_identifier' => $SingleVisitReason->visitType->identifier,
        ];
        $bulkAttributes['main_visitor_id'] = null;
        $bulkAttributes['visit_reason_id'] = $BulkAppointmentReason->id;
        $bulkAttributes['visitors'] = null;

        $BulkAppointment = $this->create($bulkAttributes, $visitorIds);

        $attributes['bulk_appointment_id'] = $BulkAppointment->id;
        $attributes['visit_reason_id'] = $SingleVisitReason->id;

        foreach ($visitorIds as $visitorId) {
            $attributes['main_visitor_id'] = $visitorId;

            $this->create($attributes);
        }
        return $BulkAppointment;

    } // createBulk()


    /**
     *
     * @param       int|string|VisitAppointment $appointment
     *
     * @return      VisitAppointment|DerivedVisitAppointment
     *
     * @version     2.1.0 / 2025-01-26
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function delete(int|string|VisitAppointment $appointment): VisitAppointment|DerivedVisitAppointment
    {
        $Action = new Delete();

        $Model = $Action->execute($appointment);

        $this->_sendNotification($Model, ModelEvent::Deleted);

        return $Model;

    } // delete()


    /**
     * Finalisiert alle Termine, deren Endzeitpunkt in der Vergangenheit liegt
     *
     * @return      int Liefert die Anzahl der betroffenen Datensätze.
     *
     * @version     1.0.0 / 2024-10-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function finishPast(): int
    {
        $Action = new FinishPast();

        return $Action->execute();

    } // finishPast()


    /**
     * Erzeugt eine Verknüpfung zwischen einem Appointment und einem Employee
     *
     * @param       int|string $appointmentId
     *
     * @param       string $employeeId
     *
     * @return      VisitAppointment|DerivedVisitAppointment
     *
     * @version     2.1.0 / 2024-10-08
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function linkEmployee(int|string $appointmentId, int|string $employeeId): VisitAppointment|DerivedVisitAppointment
    {
        /*
        **  Verknüpfung zwischen Appointment und Employee anlegen. */
        $AppointmentEmployeeService = new VisitAppointmentEmployeeService();

        return $AppointmentEmployeeService->create([
            'appointment_id' => $appointmentId,
            'employee_id' => $employeeId,
        ]);
    } // linkEmployee()


    /**
     * Erzeugt Verknüpfungen zwischen einem Appointment und den übergebenen Employees
     *
     * @param       int|string $appointmentId
     *
     * @param       array $employeeIds
     *              Ein Array mit den IDs der Employees-Daten.
     *
     * @return      array Ein Array mit Employee-Models.
     *
     * @version     2.0.0 / 2024-10-08
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function linkEmployees(int|string $appointmentId, array $employeeIds): array
    {
        $linkedEmployees = [];

        foreach ($employeeIds as $employeeId) {
            $linkedEmployees[] = $this->linkEmployee($appointmentId, $employeeId);
        }
        return $linkedEmployees;

    } // linkEmployees()


    /**
     *
     * @param       int|string|VisitAppointment|DerivedVisitAppointment $appointment
     *
     * @param       array|null $attributes
     *
     * @param       array $attributes
     *
     * @return      VisitAppointment|DerivedVisitAppointment
     *
     * @version     1.1.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function setProcessStatusCalledIn(int|string|VisitAppointment $appointment, array|null $attributes = null): VisitAppointment|DerivedVisitAppointment
    {
        $Action = new UpdateProcessStatus();

        $Appointment = $Action->execute($appointment, ProcessStatus::CalledIn, $attributes);

        return $Appointment;

    } // setProcessStatusCalledIn()


    /**
     *
     * @param       int|string|VisitAppointment|DerivedVisitAppointment $appointment
     *
     * @param       array|null $attributes
     *
     * @return      VisitAppointment|DerivedVisitAppointment
     *
     * @version     1.3.0 / 2025-03-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function setProcessStatusCheckedIn(int|string|VisitAppointment $appointment, array|null $attributes = null): VisitAppointment|DerivedVisitAppointment
    {
        $Action = new UpdateProcessStatus();

        $Appointment = $Action->execute($appointment, ProcessStatus::CheckedIn, $attributes);

        if ($Appointment->main_employee_id !== null) {
            $Employee = VisitEmployeeRepository::getById($Appointment->main_employee_id);

            if ($Employee !== null && ($Employee->email !== null || $Employee->mobile_phone_number !== null)) {
                $Employee->notify(new EmployeeVisitorArrived($Appointment));
            }
        }
        return $Appointment;

    } // setProcessStatusCheckedIn()


    /**
     *
     * @param       int|string|VisitAppointment|DerivedVisitAppointment $appointment
     *
     * @param       array|null $attributes
     *
     * @return      VisitAppointment|DerivedVisitAppointment
     *
     * @version     1.1.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function setProcessStatusCheckedOut(int|string|VisitAppointment $appointment, array|null $attributes = null): VisitAppointment|DerivedVisitAppointment
    {
        $Action = new UpdateProcessStatus();

        $Appointment = $Action->execute($appointment, ProcessStatus::CheckedOut, $attributes);

        return $Appointment;

    } // setProcessStatusCheckedOut()


    /**
     *
     * @param       int|string|VisitAppointment|DerivedVisitAppointment $appointment
     *
     * @param       array|null $attributes
     *
     * @return      VisitAppointment|DerivedVisitAppointment
     *
     * @version     1.1.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function setProcessStatusEntered(int|string|VisitAppointment $appointment, array|null $attributes = null): VisitAppointment|DerivedVisitAppointment
    {
        $Action = new UpdateProcessStatus();

        $Appointment = $Action->execute($appointment, ProcessStatus::Entered, $attributes);

        return $Appointment;

    } // setProcessStatusEntered()


    /**
     *
     * @param       int|string|VisitAppointment|DerivedVisitAppointment $appointment
     *
     * @param       array|null $attributes
     *
     * @return      VisitAppointment|DerivedVisitAppointment
     *
     * @version     1.1.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function setProcessStatusInitialized(int|string|VisitAppointment $appointment, array|null $attributes = null): VisitAppointment|DerivedVisitAppointment
    {
        $Action = new UpdateProcessStatus();

        $Appointment = $Action->execute($appointment, ProcessStatus::Initialized, $attributes);;

        return $Appointment;

    } // setProcessStatusInitialized()


    /**
     *
     * @param       int|string|VisitAppointment|DerivedVisitAppointment $appointment
     *
     * @param       array|null $attributes
     *
     * @return      VisitAppointment|DerivedVisitAppointment
     *
     * @version     1.1.0 / 2025-01-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function setProcessStatusOngoing(int|string|VisitAppointment $appointment, array|null $attributes = null): VisitAppointment|DerivedVisitAppointment
    {
        $Action = new UpdateProcessStatus();

        $Appointment = $Action->execute($appointment, ProcessStatus::Ongoing, $attributes);

        return $Appointment;

    } // setProcessStatusOngoing()


    /**
     *
     * @param       int|string|VisitAppointment|DerivedVisitAppointment $appointment
     *
     * @param       array $attributes
     *              Wenn der Schlüssel "visitors" in $attributes verwendet wird, dann wird dort ein assoziativer
     *              Array mit den Feldern "company", "email", "first_name", "last_name" und "mobile_phone_number"
     *              erwartet. Fehlende Schlüssel werden ergänzt und mit dem Wert NULL belegt.
     *              Der Schlüssel "visitors" wird ignoriert/überschrieben, wenn $visitorIDs nicht leer ist.
     *
     * @param       array $visitorIds
     *              Erwartet einen Array mit Visitor-IDs.
     *              Die Methoden ermittelt alle Visitors zu den übergebenen IDs und speichert deren Daten im
     *              Feld "visitors". Wenn dieser Parameter verwendet wird, dann wird ein eventuell in $attributes
     *              enthaltener Schlüssel "visitors" ignoriert.
     *
     * @param       bool $sendUpdateNotification
     *
     * @return      VisitAppointment|DerivedVisitAppointment
     *
     * @version     4.1.0 / 2025-02-10
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function update(int|string|VisitAppointment|DerivedVisitAppointment $appointment, array $attributes, array $visitorIds = [], bool $sendUpdateNotification = false): VisitAppointment|DerivedVisitAppointment
    {
        if ($appointment instanceof VisitAppointment) {
            $Model = $appointment;
        } else {
            $Model = VisitAppointment::find($appointment);
        }
        $VisitType = VisitTypeRepository::getByVisitReason($Model->visit_reason_id);

        if ($VisitType->identifier === VisitTypeIdentifier::BulkAppointment->value && !empty($visitorIds)) {
            $attributes['visitors'] = $this->_processBulkVisitors($visitorIds);

        } elseif ($VisitType->identifier === VisitTypeIdentifier::GroupVisit->value && !empty($attributes['visitors'])) {
            $attributes['visitors'] = $this->_processGroupVisitors($attributes['visitors']);
        }
        $Action = new Update();

        $Model = $Action->execute($Model, $attributes);

        if ($sendUpdateNotification === true) {
            $this->_sendNotification($Model, ModelEvent::Updated);
        }
        return $Model;

    } // update()


    /**
     *
     * @param       int|string|VisitAppointment $appointment
     *
     * @param       array $attributes
     *              Der Schlüssel "visitors" sollte hier nicht verwendet werden. Er wird explizit auf NULL gesetzt.
     *
     * @param       array $visitorIds
     *              Erwartet einen Array mit Visitor-IDs.
     *              Die Methode ermittelt alle Visitors zu den übergebenen IDs und speichert deren Basisdaten im
     *              Feld "visitors" des Master-Termins.
     *
     * @param       bool $sendUpdateNotification
     *
     * @return      VisitAppointment|DerivedVisitAppointment
     *
     * @version     1.2.0 / 2025-02-10
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function updateBulk(int|string|VisitAppointment $appointment, array $attributes, array $visitorIds = [], bool $sendUpdateNotification = false): VisitAppointment|DerivedVisitAppointment
    {
        if ($appointment instanceof VisitAppointment) {
            $BulkAppointment = $appointment;
        } else {
            $BulkAppointment = VisitAppointment::find($appointment);
        }
        $BulkAppointmentReason = VisitReasonRepository::getModelByIdentifier(VisitReasonDefaultIdentifier::BulkAppointment);
        /*
        **  IDs der Besucher ermitteln die bereits beim Master-Termin gespeichert sind. */
        $appointmentVisitorIds = array_map(function ($Item) {
            return (int) $Item->id;

        }, $BulkAppointment->visitors);
        /*
        **  IDs der Besucher ermitteln, die aus der Auswahl im Formular entfernt wurden. */
        $removedVisitorIds = array_diff($appointmentVisitorIds, $visitorIds);
        /*
        **  Attribut "visitors" für alle involvierten Termine zurücksetzen. Beim Speichern des Master-Termins
        **  wird das Feld mit den übergebenen IDs neu initialisiert. */
        $attributes['visitors'] = null;
        /*
        **  Übergebene Attribute für den Master-Termin aufbereiten und ein Update ausführen. */
        $bulkAttributes = $attributes;

        $bulkAttributes['main_visitor_id'] = null;
        $bulkAttributes['visit_reason_id'] = $BulkAppointmentReason->id;

        $BulkAppointment = $this->update($BulkAppointment, $bulkAttributes, $visitorIds, $sendUpdateNotification);
        /*
        **  Alle Einzeltermine ermitteln und abarbeiten. */
        $Appointments = VisitAppointmentRepository::getBulkAppointments($BulkAppointment->id);

        $visitorAppointments = [];

        foreach ($Appointments as $Appointment) {
            $visitorId = $Appointment->main_visitor_id;
            /*
            **  Termine der entfernten Besucher löschen und den Besucher dann überspringen. */
            if (in_array($visitorId, $removedVisitorIds)) {
                $this->delete($Appointment);

                continue;
            }
            /*
            **  Verbliebene Termine werden gesammelt und über die ID des jeweiligen Hauptbesuchers
            **  verfügbar gemacht. */
            $visitorAppointments[$visitorId] = $Appointment;
        }
        /*
        **  Übergebene Visitor-IDs abarbeiten und Termine aktualisieren oder erzeugen. */
        foreach ($visitorIds as $visitorId) {
            if (isset($visitorAppointments[$visitorId])) {
                unset($attributes['main_visitor_id']);

                $this->update($visitorAppointments[$visitorId], $attributes, [], $sendUpdateNotification);

            } else {
                $attributes['bulk_appointment_id'] = $BulkAppointment->id;
                $attributes['main_visitor_id'] = $visitorId;

                $this->create($attributes, ForBulkAppointment: $BulkAppointment);
            }
        }
        return $BulkAppointment;

    } // updateBulk()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class VisitAppointmentService {}
