<?php
/**
 * Update Action Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-modules/visitor-management
 * @subpackage  Actions
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanModules\VisitorManagement\Actions\VisitAppointment;


use BplanComponents\LaravelSpryng\Notifications\SmsChannel;
use BplanModules\VisitorManagement\Enums\ProcessStatus;
use BplanModules\VisitorManagement\Enums\VisitTypeGroupIdentifier;
use BplanModules\VisitorManagement\Enums\VisitTypeIdentifier;
use BplanModules\VisitorManagement\Models\VisitAppointment;
use BplanModules\VisitorManagement\Notifications\EmployeeVisitorArrived;
use BplanModules\VisitorManagement\Notifications\LogisticsVisitorArrived;
use BplanModules\VisitorManagement\Repositories\VisitEmployeeRepository;
use BplanModules\VisitorManagement\Services\VisitAppointmentStatusLogService;
use Carbon\Carbon;
use Illuminate\Support\Facades\Notification;


/**
 * Update Action Class
 *
 * @version     1.5.0 / 2025-01-26
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class UpdateProcessStatus
{

/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       VisitAppointment $Appointment
     *
     * @return 	    void
     *
     * @version     1.2.0 / 2025-01-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _sendVisitorArrivedNotification(VisitAppointment $Appointment): void
    {
        /*
        **  Benachrichtigungen zur Ankunft des Besuchers an Mitarbeiter senden. */
        if ($Appointment->visitType->visit_group === VisitTypeGroupIdentifier::Logistics->name) {
            $lang = config('visitor-management.loading-bay.notification-language');
            /*
            **  Mail und SMS an die Ladestelle.
            **
            **  @todo    Auslagern in eine Notification oder Methode. */
            Notification::route('mail', [
                        config('visitor-management.loading-bay.email') => trans('notifications.logistics-loading-bay', [], $lang),
                    ]
                )
                ->route(SmsChannel::class, config('visitor-management.loading-bay.mobile-phone-number'))
                ->notify(new LogisticsVisitorArrived($Appointment)
            );
        } else {
            /*
            **  Mail und SMS an den verantwortlichen Mitarbeiter. */
            $Employee = VisitEmployeeRepository::getById($Appointment->main_employee_id);

            $Employee->notify(new EmployeeVisitorArrived($Appointment));
        }
    } // _sendVisitorArrivedNotification()


    /**
     *
     * @todo        Es sollte noch abgeprüft und sichergestellt werden, dass ein bereits erreichter
     *              Status nicht wieder mit einem niedrigeren Wert belegt wird. Ansonsten könnte zum
     *              Beispiel eine bereits erteilte Zufahrtsgenehmigung (called in) wieder verlorengehen,
     *              wenn der Besucher den Prozess am Terminal erneut startet.
     *
     * @param       int|string|VisitAppointment $appointment
     *
     * @param       ProcessStatus $Status
     *
     * @param       array|null $attributes
     *
     * @return      VisitAppointment
     *
     * @version     1.3.0 / 2024-12-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function execute(int|string|VisitAppointment $appointment, ProcessStatus $Status, array|null $attributes = null): VisitAppointment
    {
        if ($appointment instanceof VisitAppointment) {
            $Model = $appointment;
        } else {
            $Model = VisitAppointment::findOrFail($appointment);
        }
        if ($attributes !== null) {
            $Model->fill($attributes);
        }
        $Model->process_status = $Status->value;

        $entryPointId = null;
        $Now = Carbon::now();

        switch ($Status) {
            case /* 1 */ ProcessStatus::Initialized:
                break;

            case /* 2 */ ProcessStatus::CheckedIn:
                $Model->checked_in_at = $Now;

                if ($Model->visitType->visit_group !== VisitTypeGroupIdentifier::Logistics->name) {
                    /*
                    **  Bei Nicht-Logistik-Besuchen wird von CheckIn direkt auf Entered gewechselt. */
                    return $this->execute($Model, ProcessStatus::Entered, $attributes);

                } elseif ($Model->visitType->identifier === VisitTypeIdentifier::ParcelService->name
                  || $Model->visitType->identifier === VisitTypeIdentifier::Shunting->name) {
                    /*
                    **  Bei Logistic > ParcelService und Logistic > Shunting wird beim CheckIn direkt
                    **  zu CalledIn gewechselt. */
                    return $this->execute($Model, ProcessStatus::CalledIn, $attributes);
                }
                break;

            case /* 3 */ ProcessStatus::CalledIn:
                $Model->called_in_at = $Now;
                break;

            case /* 4 */ ProcessStatus::Entered:
                $Model->entered_at = $Now;
                $entryPointId = $Model->check_in_entry_point_id;
                break;

            case /* 5 */ ProcessStatus::Ongoing:
                break;

            case /* 6 */ ProcessStatus::CheckedOut:
                /*
                **  Wenn der Termin über den aktuellen Tag hinaus gültig ist oder wenn es sich um einen
                **  ParcelService- oder Shunting-Prozess handelt, dann wird er nicht abgeschlossen.
                **  ParcelService- und Shunting-Prozesse können immer nur durch den automatischen
                **  Finish-Lauf geschlossen werden. */
                if ($Model->valid_until > Carbon::now()->endOfDay()
                  || $Model->visitType->identifier === VisitTypeIdentifier::ParcelService->name
                  || $Model->visitType->identifier === VisitTypeIdentifier::Shunting->name) {
                    $Model->checked_out_at = $Now;
                    $entryPointId = $Model->check_out_entry_point_id;

                } else {
                    $Model->checked_out_at = $Now;

                    return $this->execute($Model, ProcessStatus::Finished, $attributes);
                }
                break;

            case /* 7 */ ProcessStatus::Finished:
                $Model->finished_at = $Now;
                $entryPointId = $Model->check_out_entry_point_id;
                break;
        }
        $Model->save();

        if ($Model->visitType->visit_group === VisitTypeGroupIdentifier::Logistics->name) {
            if ($Status === ProcessStatus::CheckedIn->value) {
                $this->_sendVisitorArrivedNotification($Model);
            }
        } else {
            if ($Status === ProcessStatus::Entered->value) {
                $this->_sendVisitorArrivedNotification($Model);
            }
        }
        $LogService = new VisitAppointmentStatusLogService();

        $LogService->create([
            'appointment_id' => $Model->id,
            'entry_point_id' => $entryPointId,
            'process_status' => $Model->process_status,
        ]);

        return $Model;

    } // execute()


} // class UpdateProcessStatus {}
