<?php
/**
 * Procedure Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan/laravel-vega-rpc
 * @subpackage  Procedures
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

declare(strict_types=1);

namespace BplanModules\VisitorManagement\Http\Procedures;


use BplanModules\VisitorManagement\Repositories\VisitAppointmentRepository;
use BplanModules\VisitorManagement\Repositories\VisitEmployeeRepository;
use BplanModules\VisitorManagement\Repositories\VisitVisitorRepository;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Storage;
use Sajya\Server\Annotations\Param;
use Sajya\Server\Annotations\Result;
use Sajya\Server\Http\Request;
use Sajya\Server\Procedure;


/**
 * Procedure Class
 *
 * @version     2.9.0 / 2025-02-06
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class LayouterProcedure extends Procedure
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * The name of the procedure that is used for referencing.
     *
     * @var     string
     */
    public static string $name = 'layouter';


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Create the print-file
     *
     * @Param(name="appointmentId", value="required|string")
     *
     * @Result(name="files", value="required|array")
     *
     * @param       Request $Request
     *
     * @return      array
     *
     * @version     1.7.0 / 2025-02-06
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function create(Request $Request): array
    {
        $Storage = Storage::disk('local');

        $config = config('project.layouter');

        if($config['binary'] === null) {
            throw new Exception('No binary configured');

        } elseif (file_exists($config['binary']) === false) {
            throw new Exception('Could not find binary');

        } elseif (is_executable($config['binary']) === false) {
            throw new Exception('['.$config['binary'].'] is not executable');
        }
        $binaryPath = dirname($config['binary']);
        $executable = basename($config['binary']);
        $iniFile = $binaryPath.'/InitialCustomer/'.$config['ini-file-name'];

        if (file_exists($iniFile) === false) {
            throw new Exception('Could not find ini file '.$iniFile);
        }
        /*
        **  Alle Verzeichnisnamen und Pfade sind für die Speicherung der Queue-Datei (Json). Lediglich der
        **  Verzeichnisname in "$dirName" wird auch für die Ausgabedateien des Layouters verwendet und muss
        **  in der ini-Datei genutzt werden. */
        $dirName = 'layouter';
        $subPath = 'queue/'.$dirName;

        $targetPath = $Storage->path($subPath);

        if ($Storage->exists($subPath) === false) {
            if (mkdir($targetPath, 0777, true) === false) {
                throw new Exception('Could not create target directory');
            }

        } elseif (is_writable($targetPath) === false) {
            throw new Exception('Target directory is not writable');
        }
        $requestParams = $Request->getParams()->toArray();

        $Appointment = VisitAppointmentRepository::getById($requestParams['appointmentId']);

        $Employee = null;

        if ($Appointment->main_employee_id !== null) {
            $Employee = VisitEmployeeRepository::getById($Appointment->main_employee_id);
        }
        $Visitor = VisitVisitorRepository::getById($Appointment->main_visitor_id);

        $checkIn = Carbon::now()->setTimezone($config['timezone'])->addHours($config['timezone-offset'])->format($config['date-time-format']);
        $validFrom = $Appointment->valid_from->setTimezone($config['timezone'])->addHours($config['timezone-offset'])->format($config['date-time-format']);
        $validUntil = $Appointment->valid_until->setTimezone($config['timezone'])->addHours($config['timezone-offset'])->format($config['date-time-format']);

        $fileData = [
            'visitors' => [
                [
                    'id' => $Appointment->id,
                    'checkIn' => $checkIn,
                    'company' => $Visitor->company,
                    'firstName' => $Visitor->first_name,
                    'fullName' => $Visitor->getFullName(reverted: true),
                    'lastName' => $Visitor->last_name,
                    'licensePlate' => $Appointment->license_plate,
                    'hostFirstName' => ($Employee === null ? '' : $Employee->first_name),
                    'hostFullName' => ($Employee === null ? '' : $Employee->getFullName()),
                    'hostLastName' => ($Employee === null ? '' : $Employee->last_name),
                    'token' => $Appointment->token,
                    'validFrom' => $validFrom,
                    'validUntil' => $validUntil,
                ],
            ]
        ];
        $fileName = $Appointment->token.'.json';
        $queueFile = $Storage->path($subPath.'/'.$fileName);

        if ($Storage->put($subPath.'/'.$fileName, json_encode($fileData, JSON_PRETTY_PRINT)) === false) {
            throw new Exception('Could not create file');
        }
        $command = '"'.$executable.'" '.$config['ini-file-name'].' "'.$queueFile.'"';

        $cwd = getcwd();
        /*
        **  Ins Verzeichnis des Layouters wechseln. */
        chdir($binaryPath);

        ob_start(); {
            $lastOutputLine = system($command, $resultCode);

        } ob_end_clean();

        chdir($cwd);

        if($resultCode !== false) {
            switch ($resultCode) {
                case 1:
                    throw new Exception('Error while creating: error code '.$resultCode.', '.$lastOutputLine);
                    break;

                case 0:
                    if(!empty($lastOutputLine)) {
                        throw new Exception('Error while creating: error code '.$resultCode.', '.$lastOutputLine);
                    }
                    // no break
                default:
                    // Success
            }
        }
        try {
            unlink($queueFile);

        } catch (\Throwable $th) {
            //
        }
        /*
        **  Dateien für das Ergebnis zusammenstellen. */
        $files = [];
        $storageConfig = Storage::disk('public')->getConfig();

        foreach ($fileData['visitors'] as $key => $nul) {
            $files[] =  $storageConfig['url'].'/'.$dirName.'/'.$key.'_'.$config['output-file-name'].'.'.$config['output-file-extension'];
        }
        return [
            'files' => $files,
        ];
    } // create()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class LayouterProcedure extends Procedure {}
