<?php
/**
 * Import Class
 *
 * @todo        Es muss geprüft werden wie der Hauptbesucher identifziert werden kann.
 *              Er sollte nicht in der Importdatei enthalten sein und entfernt werden, falls er doch enthalten ist.
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-modules/visitor-management
 * @subpackage  Imports
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanModules\VisitorManagement\Imports;


use BplanBase\Globals\Helpers\StringHelper;
use BplanModules\VisitorManagement\Enums\ImportType;
use BplanModules\VisitorManagement\Enums\ProcessStatus;
use BplanModules\VisitorManagement\Repositories\VisitAppointmentRepository;
use BplanModules\VisitorManagement\Services\VisitAppointmentService;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\Importable;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Events\AfterImport;
use Maatwebsite\Excel\Events\BeforeImport;


/**
 * Import Class
 *
 * @version     4.3.0 / 2025-04-29
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class GroupVisitorsImport extends BaseImport implements
    ToCollection,
    WithValidation
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    use Importable;


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Die ID des Appointments
     *
     * @var     int $_appointment_id
     */
    private $_appointment_id;


    /**
     * Objekt der Service-Klasse zum Speichern der Visitors zum Appointment
     *
     * @var     VisitAppointmentService $_AppointmentService
     */
    private $_AppointmentService;


    /**
     * @var     ImportType $_ImportType
     */
    protected $_ImportType = ImportType::GroupVisitor;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string $_subPath
     */
    protected static $_subPath = 'import/group-visitors';


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Führt Aktionen nach dem Import aus
     *
     * @param       AfterImport $Event
     *
     * @version     1.0.0 / 2024-09-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function afterImport(AfterImport $Event): void
    {
        parent::afterImport($Event);

    } // afterImport()


    /**
     * Führt Aktionen vor dem Import aus
     *
     * @param       BeforeImport $Event
     *
     * @return      int
     *
     * @version     1.1.0 / 2024-10-23
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function beforeImport(BeforeImport $Event): void
    {
        parent::beforeImport($Event);

        $this->_AppointmentService = new VisitAppointmentService();

    } // beforeImport()


    /**
     *
     * @return 	    void
     *
     * @version     1.2.0 / 2025-02-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function collection(Collection $Rows)
    {
        $visitors = [];

        foreach ($Rows as $row) {
            $mobilePhoneNumber = StringHelper::explodePhoneNumber($row['mobile_phone_number']);

            $visitors[] = [
                // 'active'                           => 1,
                'company'                          => $row['company'],
                'email'                            => $row['email'],
                'first_name'                       => $row['first_name'],
                'last_name'                        => $row['last_name'],
                'mobile_phone_number'              => $mobilePhoneNumber['phone-number'],
                'mobile_phone_number_country_code' => $mobilePhoneNumber['country-code'],
            ];
            $this->_incrementRowCountSaved();
        }
        $this->_AppointmentService->update($this->_appointment_id, ['visitors' => $visitors]);

    } // collection()


    /**
     * Bereitet Werte einer Zeile vor der Validierung auf
     *
     * Als Schlüsselnamen müssen die aufbereiteten Spaltennamen (First Name > first_name) aus der Importdatei
     * verwendet werden.
     *
     * @param       array $data
     *              Die aktuelle Zeile als Array.
     *
     * @param       int $index
     *              Die Zeilennummer.
     *
     * @return      array
     *
     * @requires    Maatwebsite\Excel\Concerns\WithValidation
     *              Diese Methode wird automatisch in der Methode Maatwebsite\Excel\Row::toArray() aufgerufen. Das führt
     *              leider dazu, dass der Code jedes Mal ausgeführt wird, wenn toArray() verwendet wird. Da dabei aber
     *              immer das unveränderte Row-Objekt verwendet wird, kommt es glücklicherweise nicht dazu, dass Daten
     *              doppelt modifiziert werden.
     *
     * @version     1.2.0 / 2025-01-30
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function prepareForValidation($data, $index): array
    {
        $data = parent::prepareForValidation($data, $index);

        $data['email'] = StringHelper::normalizeEmail($data['email']);
        $data['mobile_phone_number'] = StringHelper::normalizePhoneNumber($data['mobile_phone_number']);

        return $data;

    } // prepareForValidation()


    /**
     * Liefert ValidationRules
     *
     * In dieser Methode müssen die aufbereiteten Spaltennamen (First Name > first_name) aus der Importdatei
     * als Schlüssel verwendet werden.
     *
     * @return      array
     *
     * @requires    Maatwebsite\Excel\Concerns\WithValidation
     *              Damit die Methode rules() automatisch verwendet wird.
     *
     * @requires    Maatwebsite\Excel\Concerns\WithHeadingRow
     *              Um die Überschriften der Spalten als Schlüssel und auch in den Rules (z.B. "maximum" => "gte:*.minimum")
     *              verwenden zu können.
     *
     * @version     1.1.0 / 2024-10-23
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function rules(): array
    {
        return [
            'company'             => ['nullable', 'max:100'],
            'email'               => ['nullable', 'max:255', 'email'],
            'first_name'          => ['required', 'max:50'],
            'last_name'           => ['required', 'max:50'],
            'mobile_phone_number' => ['nullable', 'max:20'],
        ];
    } // rules()


    /**
     *
     * @param       int $appointment_id
     *
     * @return 		$this
     *
     * @version		1.2.0 / 2025-04-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function setAppointmentID(int $appointment_id): self
    {
        $this->_appointment_id = $appointment_id;

        $Appointment = VisitAppointmentRepository::getModelByID($appointment_id, ignoreFinishedStatus: true);

        $Timestamp = $Appointment->valid_until;
        /*
        **  Sicherstellen, dass nur VisitAppointments aktualisiert werden die nicht bereits abgeschlossen sind. */
        if ($Timestamp->lt(Carbon::now()) || (int) $Appointment->process_status === ProcessStatus::Finished->value) {
            throw new Exception(__('visitor-management::group-visitor-import.error.appointment-already-finished', [
                'id' => $Appointment->id
            ]));
        }
        return $this;

    } // setAppointmentID()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class GroupVisitorsImport extends BaseImport implements ... {}
