<?php
/**
 * Livewire Component Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-modules/visitor-management
 * @subpackage  LivewireComponents
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanModules\VisitorManagement\Livewire\VisitAppointments\Modals;


use BplanBase\Globals\Enums\LogLevel;
use BplanBase\Globals\Enums\RequireStatus;
use BplanBase\Globals\Services\LogService;
use BplanBase\GlobalComponents\Traits\WithComboBoxInput;
use BplanModules\VisitorManagement\Enums\AppointmentSteps;
use BplanModules\VisitorManagement\Enums\VisitTypeGroupIdentifier;
use BplanModules\VisitorManagement\Models\VisitEmployee;
use BplanModules\VisitorManagement\Models\VisitVisitor;
use BplanModules\VisitorManagement\Models\VisitReason;
use BplanModules\VisitorManagement\Repositories\VisitEmployeeRepository;
use BplanModules\VisitorManagement\Repositories\VisitReasonRepository;
use BplanModules\VisitorManagement\Services\VisitAppointmentService;
use BplanModules\VisitorManagement\Services\VisitVisitorService;
use BplanModules\VisitorManagement\Traits\AppointmentPagination;
use BplanModules\VisitorManagement\Traits\WithLicensePlateInput;
use Illuminate\Database\Eloquent\Collection;
use Laravel\Jetstream\InteractsWithBanner;
use Livewire\Attributes\On;
use Livewire\Component;
use stdClass;
use Illuminate\Support\Str;


/**
 * Livewire Component Class
 *
 * @version     1.1.0 / 2025-04-29
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class Create extends Component
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    use InteractsWithBanner;
    use WithLicensePlateInput;
    use WithComboBoxInput;
    use AppointmentPagination;


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     */
    public ?Collection $_allVisitors;


    /**
     *
     */
    // #[Validate('required')]
    public bool $active = true;


    /**
     *
     */
    public ?string $appointmentBegin;


    /**
     *
     */
    // #[Validate('after:appointmentBegin')]
    public ?string $appointmentEnd;


    /**
     *
     */
    protected VisitAppointmentService $appointmentService;


    /**
     * Wird für WithComboBoxInput benötigt
     */
    public ?array $comboBoxData;


    /**
     *
     */
    // #[Validate('required_if:isSpontaneousVisit,false')]
    public bool $driveOnAreaPermitted = false; //* Kennzeichen erfassen


    /**
     *
     */
    public ?Collection $employees;


    /**
     *
     */
    // #[Validate('required')]
    public ?string $endDate;


    /**
     *
     */
    // #[Validate('required')]
    public ?string $endTime;


    /**
     *
     */
    public bool $internalVisit = false;


    /**
     *
     */
    public bool $isBulkVisit = false;


    /**
     *
     */
    // #[Validate('required')]
    public bool $isSpontaneousVisit = false;


    /**
     *
     */
    public ?array $filteredComboBoxData;


    /**
     * Nur wenn driveOnAreaPermitted = true || VisitType == AccessAuthorization
     */
    // #[Validate('required')]
    public string $licensePlateFormatted = '';


    /**
     *
     */
    protected LogService $logService;


    /**
     *
     */
    // #[Validate('nullable|required_if:isSpontaneousVisit,false|exists:employees,id')] // nullable|numeric|
    public ?int $mainEmployeeId = null;


    /**
     *
     */
    // #[Validate('nullable|required_if:isSpontaneousVisit,false|exists:visitors,id')] // nullable|numeric|
    public ?int $mainVisitorId = null;


    /**
     *
     */
    // #[Validate('required')]
    public ?string $orderNumber = null;


    /**
     *
     */
    public ?int $organizerEmployeeId = null;


    /**
     *
     */
    // #[Validate('required_if:isSpontaneousVisit,false')]
    public bool $requireEscort = false;


    /**
     *
     */
    // #[Validate('required')]
    public ?string $startDate;


    /**
     *
     */
    // #[Validate('required')]
    public ?string $startTime;


    /**
     *
     */
    // #[Validate('required')]
    public ?string $subject;


    /**
     *
     */
    // #[Validate('required')]
    public ?array $visitors;


    /**
     *
     */
    // #[Validate('required|exists:visit_reasons,id')]
    public ?int $visitReasonId = null; // Abhängig zu visitType


    /**
     *
     */
    public ?array $visitReasonsByVisitType;


    /**
     *
     */
    public ?stdClass $visitReasonLabels;


    /**
     *
     */
    public ?Collection $visitReasons;


    /**
     *
     */
    // #[Validate('required_if:isSpontaneousVisit,false')]
    public bool $withMinors = false;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _setMainEmployee()
    {
        $mainEmployee = array_filter(
            array_map(function ($item) {
                if (isset($item['isSelected']) && $item['isSelected']) {
                    return $item['id'];
                }
            }, $this->comboBoxData['mainEmployees'])
        );
        if (isset($mainEmployee) && !empty($mainEmployee)) {
            $this->mainEmployeeId = $mainEmployee[array_key_first($mainEmployee)];
        }
    } // _setMainEmployee()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _setOrganizer()
    {
        $organizerEmployee = array_filter(
            array_map(function ($item) {
                if (isset($item['isSelected']) && $item['isSelected']) {
                    return $item['id'];
                }
            }, $this->comboBoxData['organizers'])
        );

        if (isset($organizerEmployee) && !empty($organizerEmployee)) {
            $this->organizerEmployeeId = $organizerEmployee[array_key_first($organizerEmployee)];
        }
    } // _setOrganizer()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _setMainVisitor()
    {
        $mainVisitor = array_filter(
            array_map(function ($item) {
                if (isset($item['isSelected']) && $item['isSelected']) {
                    return $item['id'];
                }
            }, $this->comboBoxData['mainVisitors'])
        );
        if (isset($mainVisitor) && !empty($mainVisitor)) {
            $this->mainVisitorId = $mainVisitor[array_key_first($mainVisitor)];
        }
    } // _setMainVisitor()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _setVisitors()
    {
        $visitors = array_filter(
            array_map(function ($item) {
                if (isset($item['isSelected']) && $item['isSelected']) {
                    return $item['id'];
                }
            }, $this->comboBoxData['visitors'])
        );
        if (isset($visitors) && !empty($visitors)) {
            $this->visitors = $visitors;
        }
    } // _setVisitors()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function boot(LogService $logService, VisitAppointmentService $appointmentService)
    {
        $this->logService = $logService;
        $this->appointmentService = $appointmentService;

    } // boot()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function buildAppointmentEnd()
    {
        $this->appointmentEnd = $this->endDate . 'T' . $this->endTime;

    } // buildAppointmentEnd()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function buildAppointmentStart()
    {
        $this->appointmentBegin = $this->startDate . 'T' . $this->startTime;

    } // buildAppointmentStart()


    /**
     *
     * @version     1.1.0 / 2025-04-29
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function createAppointment($confirmed = false)
    {
        $this->buildAppointmentStart();
        $this->buildAppointmentEnd();

        $this->validate();

        if ($this->isSpontaneousVisit && !$confirmed && !isset($this->mainVisitorId)) {
            $this->dispatch('show-confirm-spontaneous-visit-without-visitor');
            return;
        } elseif ($this->isSpontaneousVisit && $confirmed) {
            $visitorService = new VisitVisitorService();
            $visitor = $visitorService->create([]);
            $this->mainVisitorId = $visitor->id;
        }
        try {
            $selectedVisitors = array_filter(
                array_map(function ($item) {
                    if (isset($item['isSelected']) && $item['isSelected']) {
                        return $item['id'];
                    }
                }, $this->comboBoxData['visitors'])
            );
            $attributes = [
                'active' => $this->active,
                'drive_on_area_permitted' => $this->driveOnAreaPermitted,
                'internal_visit' => $this->internalVisit,
                'license_plate_formatted' => $this->licensePlateFormatted,
                'main_visitor_id' => $this->mainVisitorId,
                'main_employee_id' => $this->mainEmployeeId,
                'order_number' => $this->orderNumber,
                'organizer_employee_id' => $this->organizerEmployeeId,
                'spontaneous_visit' => $this->isSpontaneousVisit,
                'require_escort' => $this->requireEscort,
                'subject' => $this->subject,
                'visit_reason_id' => $this->visitReasonId,
                'valid_from' => $this->appointmentBegin,
                'valid_until' => $this->appointmentEnd,
                'with_minors' => $this->withMinors,
            ];
            $appointment = null;

            if ($this->isBulkVisit) {
                $appointment = $this->appointmentService->createBulk($attributes, $selectedVisitors);

            } else {
                $appointment = $this->appointmentService->create($attributes, $selectedVisitors);
            }
            // Main Visitor zusätzlich verlinken
            // $visitorsIds[] = ['id' => $this->mainVisitorId];

            $this->logService->log(LogLevel::Info, "Appointment {$this->subject} added by " . auth()->user()->name . ' (' . auth()->user()->email . ')');

            $this->dispatch('init-tippyjs');
            $this->dispatch('appointment-created');
            $this->banner(__('visitor-management::visit-appointments.appointment_created_successfully', [
                'subject' => $this->subject
            ]));

        } catch (\Throwable $th) {
            $this->logService->log(LogLevel::Error, 'Appointments/Modals/createAppointment error: ' . $th->getMessage() . ', file:' . $th->getFile() . ', line:' .  $th->getLine());
            $this->dangerBanner('Error when creating appointment. Check the logs for more information.');
        }
    } // createAppointment()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    #[On('create-appointment')]
    public function createAppointmentWithouVisitor()
    {
        $this->createAppointment(true);

    } // createAppointmentWithouVisitor()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function mount()
    {
        $this->startDate = now()->toDateString();
        $this->startTime = now()->toTimeString();
        $this->endDate = now()->endOfDay()->toDateString();
        $this->endTime = now()->endOfDay()->toTimeString();

        $this->buildAppointmentStart();
        $this->buildAppointmentEnd();

        $this->employees = VisitEmployeeRepository::getAll();
        // $this->mainEmployeeId = isset($this->employees[0]) ? $this->employees[0]->id : null;
        $this->_allVisitors = VisitVisitor::where('verified', 1)->where('anonymized', 0)->get();
        // $this->mainVisitorId = isset($this->_allVisitors[0]) ? $this->_allVisitors[0]->id : null;
        /*
        **  VisitReasons ermitteln und dabei alle Reasons ignorieren deren VisitType "internal = 1" ist
        **  oder die selbst "internal = 1" sind. */
        $this->visitReasons = VisitReasonRepository::getAll();

        //->mapWithKeys(function ($item) {
        //    return [$item['id'] => $item];
        //});
        foreach ($this->visitReasons as $key => $visitReason) {
            $this->visitReasonsByVisitType[$visitReason->visitType->identifier][] = $visitReason;
        }
        // $this->visitReasonId = $this->visitReasons->first() != null ? $this->visitReasons->first()->id : null;

        $this->setupComboBoxes();

    } // mount()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function render()
    {
        return view('visitor-management::livewire.visit-appointments.modals.create');

    } // render()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function resetModal()
    {
        $this->clearLicensePlates();

        $this->isSpontaneousVisit = false;
        $this->subject = null;
        $this->orderNumber = null;
        $this->mainVisitorId = null;
        $this->visitReasonId = null;
        $this->mainEmployeeId = null;
        $this->appointmentBegin = null;
        $this->appointmentEnd = null;
        $this->driveOnAreaPermitted = false;
        $this->withMinors = false;
        $this->requireEscort = false;
        $this->licensePlateFormatted = false;

        $this->active = true;
        $this->internalVisit = false;

        $this->comboBoxDataDeselectAll('visitors');
        $this->comboBoxDataDeselectAll('mainVisitors');
        $this->comboBoxDataDeselectAll('mainEmployees');
        $this->comboBoxDataDeselectAll('organizers');

    } // resetModal()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function rules()
    {
        $rules = [];

        if (!isset($this->startDate)) {
            $this->startDate = now()->toDateString();
        }
        if (!isset($this->startTime)) {
            $this->startTime = now()->toTimeString();
        }
        if (!isset($this->endDate)) {
            $$this->endDate = now()->endOfDay()->toDateString();
        }
        if (!isset($this->endTime)) {
            $this->endTime = now()->endOfDay()->toTimeString();
        }
        $this->buildAppointmentStart();
        $this->buildAppointmentEnd();

        if ($this->appointmentStep >= AppointmentSteps::General->value) {
            $rules['isSpontaneousVisit'] = 'required';
            $rules['subject'] = 'required';
            $rules['appointmentBegin'] = 'required';
            $rules['appointmentEnd'] = 'required|after:appointmentBegin';
            $rules['visitReasonId'] = 'required|exists:visit_reasons,id';
        }
        if ($this->appointmentStep >= AppointmentSteps::Visitors->value) {
            $this->_setMainEmployee();
            $this->_setMainVisitor();
            $this->_setOrganizer();
            $this->_setVisitors();

            //* Verantwortlicher Mitarbeiter immer optional - so von Avient gewünscht
            // $rules['mainEmployeeId'] = 'nullable|required_if:isSpontaneousVisit,false|exists:employees,id';
            //$rules['licensePlateFormatted'] = 'nullable|required_if:driveOnAreaPermitted,true';

            if ($this->isBulkVisit) {
                $rules['visitors'] = 'required';
            }
            $visitReason = $this->visitReasons->where('id',$this->visitReasonId)->first();
            $this->visitReasonLabels = $visitReason?->labels;

            if (isset($this->visitReasonId) && isset($visitReason?->visitType?->dependencies)) {
                $visitReason = $this->visitReasons->where('id',$this->visitReasonId)->first();

                foreach ($visitReason->visitType->dependencies as $column => $required) {
                    if ($required >= RequireStatus::Required->value) {
                        // if ($column == 'license_plate_formatted')
                        // {
                        //     if ($this->isSpontaneousVisit)
                        //     {
                        //        continue;
                        //     }

                        //     $column = 'licensePlates';
                        // }
                        switch ($column) {
                            case 'license_plate_formatted':
                                if ($this->driveOnAreaPermitted === false || $this->isSpontaneousVisit) {
                                    continue 2;
                                }
                                break;
                            case 'main_visitor_id':
                                if ($this->isBulkVisit == false) {
                                    $rules['mainVisitorId'] = 'nullable|required_if:isSpontaneousVisit,false|exists:visitors,id';
                                }
                                break;

                            default:
                                // ? In den dependencies sind die spalten in snake case.
                                // ? So wird aus z.B. "main_visitor_id" => "mainVisitorId"
                                // ? Welches dann der Variablen namen in dieser Component entspricht.
                                $columnCamelCase = Str::camel($column);

                                $rules[$columnCamelCase] = 'required';
                                break;
                        }
                    }
                }
            }
        }
        return $rules;

    } // rules()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function setupComboBoxes()
    {
        $this->comboBoxData['visitors'] = $this->_allVisitors->map(function ($visitor) {
            $visitorArray = $visitor->getAttributes();
            $visitorArray['label'] = "{$visitor->last_name}, {$visitor->first_name}";
            $visitorArray['tag'] = "{$visitor->email}";

            return $visitorArray;

        })->toArray();

        $this->comboBoxData['mainVisitors'] = $this->_allVisitors->map(function ($visitor) {
            $visitorArray = $visitor->getAttributes();
            $visitorArray['label'] = "{$visitor->last_name}, {$visitor->first_name}";
            $visitorArray['tag'] = "{$visitor->email}";

            return $visitorArray;

        })->toArray();

        $this->comboBoxData['mainEmployees'] = $this->employees->map(function ($employee) {
            $visitorArray = $employee->getAttributes();
            $visitorArray['label'] = "{$employee->last_name}, {$employee->first_name}";
            $visitorArray['tag'] = "{$employee->email}";

            return $visitorArray;

        })->toArray();

        $this->comboBoxData['organizers'] = $this->employees->whereNotNull('email')->map(function ($employee) {
            $visitorArray = $employee->getAttributes();
            $visitorArray['label'] = "{$employee->last_name}, {$employee->first_name}";
            $visitorArray['tag'] = "{$employee->email}";

            return $visitorArray;

        })->toArray();

        $this->filteredComboBoxData = $this->comboBoxData;

    } // setupComboBoxes()


    /**
     *
     * @version     1.0.0 / 2025-04-23
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function updated($property)
    {
        switch ($property) {
            case 'startDate':
            case 'startTime':
                $this->buildAppointmentStart();
                break;

            case 'endDate':
            case 'endTime':
                $this->buildAppointmentEnd();
                break;

            case 'internalVisit':
                if ($this->internalVisit) {
                    $this->requireEscort = false;
                }
                break;

            case 'requireEscort':
                if ($this->requireEscort) {
                    $this->internalVisit = false;
                }
                break;

            case 'isBulkVisit':
                if ($this->isBulkVisit) {
                    $this->comboBoxDataDeselectAll('mainVisitors');
                    $this->mainVisitorId = null;

                } else {
                    $this->comboBoxDataDeselectAll('visitors');
                }
                break;

            case 'visitReasonId':
                $visitReason = $this->visitReasons->where('id', $this->visitReasonId)->first();

                if (isset($visitReason)) {
                    if ($visitReason->visitType->visit_group == VisitTypeGroupIdentifier::Logistics->value) {
                        $this->driveOnAreaPermitted = true;
                    }
                }
                break;

            default:
                # code...
                break;
        }
    } // updated()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class Create extends Component {}
