<?php
/**
 * Livewire Component
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-modules/visitor-management
 * @subpackage  LivewireComponents
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanModules\VisitorManagement\Livewire\VisitReasons\Modals;


use BplanBase\Globals\Enums\LogLevel;
use BplanBase\Globals\Services\LogService;
use BplanModules\VisitorManagement\Models\VisitReason;
use BplanModules\VisitorManagement\Models\VisitType;
use BplanModules\VisitorManagement\Services\VisitReasonService;
use Illuminate\Database\Eloquent\Collection;
use Laravel\Jetstream\InteractsWithBanner;
use Livewire\Attributes\On;
use Livewire\Attributes\Rule;
use Livewire\Attributes\Validate;
use Livewire\Component;


/**
 * Livewire Component
 *
 * @version     1.0.0 / 2025-04-13
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class Edit extends Component
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    use InteractsWithBanner;


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @var array|null
     */
    #[Rule('required')]
    public ?array $labelsJson;


    /**
     *
     * @var LogService
     */
    protected LogService $logService;


    /**
     *
     * @var VisitReason|null
     */
    public ?VisitReason $visitReason;


    /**
     *
     * @var VisitReasonService
     */
    protected VisitReasonService $visitReasonService;


    /**
     *
     * @var integer|null
     */
    #[Validate('required|exists:visit_types,id')]
    public ?int $visitTypeId;


    /**
     *
     * @var Collection
     */
    public Collection $visitTypes;


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @version     1.0.0 / 2025-04-13
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function boot(LogService $logService, VisitReasonService $visitReasonService)
    {
        $this->logService = $logService;
        $this->visitReasonService = $visitReasonService;

    } // boot()


    /**
     *
     * @version     1.0.0 / 2025-04-13
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function deleteVisitReason()
    {
        try
        {
            $this->visitReasonService->delete($this->visitReason->id);

            $this->dispatch('visit-reason-deleted');
        } catch (\Throwable $th) {
            $this->logService->log(LogLevel::Error, 'VisitReaonss/Modals/deleteVisitReaons error: ' . $th->getMessage() . ', file:' . $th->getFile() . ', line:' .  $th->getLine());
            $this->dangerBanner('Error when deleting visit reason. Check the logs for more information.');
        }
    } // deleteVisitReason()


    /**
     *
     * @version     1.0.0 / 2025-04-13
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function editVisitReason()
    {
        $this->validate();

        try {
            $this->visitReasonService->create([
                'labels' => $this->labelsJson,
                'visit_type_id' => $this->visitTypeId,
            ]);

            $this->logService->log(LogLevel::Info, "VisitReason edited by " . auth()->user()->name . ' (' . auth()->user()->email . ')');

            $this->dispatch('init-tippyjs');
            $this->dispatch('visit-reason-edited');

        } catch (\Throwable $th) {
            $this->logService->log(LogLevel::Error, 'VisitReasons/Modals/editVisitReason error: ' . $th->getMessage() . ', file:' . $th->getFile() . ', line:' .  $th->getLine());
            $this->dangerBanner('Error when editing VisitReason. Check the logs for more information.');
        }
    } // editVisitReason()


    /**
     *
     * @version     1.0.0 / 2025-04-13
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    #[On('show-edit-visit-reason-modal')]
    public function initVisitReason(VisitReason $visitReason)
    {
        $this->visitReason = $visitReason;
        $this->labelsJson = (array)$visitReason->labels;
        $this->visitTypeId = $visitReason->visit_type_id;

    } // initVisitReason()


    /**
     *
     * @version     1.0.0 / 2025-04-13
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function mount()
    {
        $this->visitTypes = VisitType::where('internal', 0)->get();

    } // mount()


    /**
     *
     * @version     1.0.0 / 2025-04-13
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    #[On('multi-language-json-created')]
    public function onMultiLanguageJsonCreated(?array $json, string $target)
    {
        if ($target == 'edit') {
            $this->labelsJson = $json;
        }
    } // onMultiLanguageJsonCreated()


    /**
     *
     * @version     1.0.0 / 2025-04-13
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function render()
    {
        return view('visitor-management::livewire.visit-reasons.modals.edit');

    } // render()


    /**
     *
     * @version     1.0.0 / 2025-04-13
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function resetModal()
    {
        $this->visitReason = null;
        $this->labelsJson = [];

    } // resetModal()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class Edit extends Component {}
