<?php
/**
 * Notification Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-modules/visitor-management
 * @subpackage  Notifications
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanModules\VisitorManagement\Notifications;


use App\Repositories\UserRepository;
use BplanComponents\ICS\ICS\Event;
use BplanComponents\ICS\ICS\ICSFile;
use BplanModules\VisitorManagement\Models\VisitAppointment;
use BplanModules\VisitorManagement\Notifications\Messages\PHPMailerMessage;
use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;


/**
 * Notification Class
 *
 * Benachrichtigung an einen Mitarbeiter (verantwortlicher Mitarbeiter) über die Löschung einer
 * Zufahrtsberechtigung (per E-Mail) mit angehängter ics-Datei.
 *
 * @version     1.0.0 / 2025-01-26
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class EmployeeAccessAuthorizationDeleted extends Notification
{

/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    use Queueable;


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     VisitAppointment $_Appointment
     */
    private $_Appointment;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Create a new notification instance.
     *
     * @param       VisitAppointment $Appointment
     *
     * @version     1.0.0 / 2025-01-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(VisitAppointment $Appointment)
    {
        $this->_Appointment = $Appointment;

    } // __construct()


    /**
     *
     * @param       VisitEmployee $Notifiable
     *
     * @param       null|string & $icsContents
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2025-01-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _buildIcsFile(object $Notifiable, null|string & $icsContents = null): string
    {
        $lang = 'de';

        $company = config('project.company');

        $CreateUser = UserRepository::getById($this->_Appointment->create_user_id);
        /*
        **  Event für ics-Datei zusammenstellen.
        **
        **  Da die verwendete ICS-Klasse keine Möglichkeit bietet eine Timezone zu definieren, muss die
        **  Timezone für Beginn und Ende des Events nach "UTC" geändert werden. */
        $eventDetails = [
            'description' => trans('notifications.access-authorization.mail.description', [
                    'license-plate' => $this->_Appointment->subject
                ], $lang
            ),
            'dtstart' => $this->_Appointment->valid_from->setTimezone('UTC')->format('Y-m-d H:i'),
            'dtend' => $this->_Appointment->valid_until->setTimezone('UTC')->format('Y-m-d H:i'),
            'location' => $company['full_name'].', '.$company['address'],
            'sequence' => 0,
            'summary' => trans('notifications.access-authorization.mail.summary', [
                    'license-plate' => $this->_Appointment->subject
                ], $lang
            ),
            'uid' => $this->_Appointment->uuid,
        ];
        $Event = new Event($eventDetails);
        /*
        **  Organizer ist immer der erstellende User und der erste Attendee ist der Mitarbeiter. */
        $Event->setOrganizer($CreateUser->name, $CreateUser->email);
        $Event->addAttendee($Notifiable->getFullName(), $Notifiable->email, accepted: true);
        /*
        **  ics-Datei erstellen und speichern. */
        $IcsFile = new ICSFile();

        $IcsFile->addEvent($Event);
        $IcsFile->setLanguage($lang);
        $IcsFile->setMethodCancel();

        $storagePath = storage_path('app/tmp');

        if (!file_exists($storagePath)) {
            mkdir($storagePath, 0777, true);
        }
        $icsFile = $storagePath.'/'.$this->_Appointment->uuid.'.ics';
        $icsContents = $IcsFile->toString();

        file_put_contents($icsFile, $icsContents);

        return $icsFile;

    } // _buildIcsFile()


    /**
     * Get the array representation of the notification.
     *
     * @param       object $Notifiable
     *
     * @return      array<string, mixed>
     *
     * @version     1.0.0 / 2025-01-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function toArray(object $Notifiable): array
    {
        return [
            //
        ];
    } // toArray()


    /**
     * Get the mail representation of the notification.
     *
     * @param       object $Notifiable
     *
     * @return      PHPMailerMessage
     *
     * @version     1.0.0 / 2025-01-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function toPHPMailer(object $Notifiable): PHPMailerMessage
    {
        $lang = 'de';

        $icsFile = $this->_buildIcsFile($Notifiable, $icsContents);

        $Message = new PHPMailerMessage();

        $Message->addRecipient($Notifiable->email, $Notifiable->getFullName())
            ->addAttachment($icsFile, 'invite.ics')
            ->setIcal($icsContents)
            ->setSubject(trans('notifications.access-authorization-deleted.mail.subject', [
                    'license-plate' => $this->_Appointment->subject
                ], $lang)
            );

        return $Message;

    } // toPHPMailer()


    /**
     * Get the notification's delivery channels.
     *
     * @param       object $Notifiable
     *
     * @return      array<int, string>
     *
     * @version     1.0.0 / 2025-01-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function via(object $Notifiable): array
    {
        return [\BplanModules\VisitorManagement\Channels\PHPMailerChannel::class];

    } // via()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class EmployeeAccessAuthorizationDeleted extends Notification {}
